package common

import (
	"crypto/tls"
	"fmt"
	"github.com/jordan-wright/email"
	"net"
	"net/smtp"
	"oa-server/app/hrcenter/hr_rpc/internal/config"
	"strings"
)

func SendEmail(emailInfo config.Email, to []string, subject string, body string) error {
	from := emailInfo.From
	nickname := emailInfo.Nickname
	secret := emailInfo.Secret
	host := emailInfo.Host
	port := emailInfo.Port
	isSSL := emailInfo.IsSSl

	auth := smtp.PlainAuth("", from, secret, host)

	e := email.NewEmail()
	if nickname != "" {
		e.From = fmt.Sprintf("%s <%s>", nickname, from)
	} else {
		e.From = from
	}
	e.To = to
	e.Subject = subject
	e.HTML = []byte(body)

	hostAddr := fmt.Sprintf("%s:%d", host, port)

	//// 设置超时上下文
	//ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	//defer cancel()

	// 根据是否使用SSL/TLS选择发送方式
	var err error
	if isSSL {
		tlsConfig := &tls.Config{
			ServerName:         host,
			MinVersion:         tls.VersionTLS12, // 要求TLS 1.2及以上
			InsecureSkipVerify: false,            // 不跳过证书验证
		}
		err = e.SendWithTLS(hostAddr, auth, tlsConfig)
	} else {
		// 非SSL方式发送
		err = e.Send(hostAddr, auth)
	}

	// 处理特定错误类型
	if err != nil {
		if netErr, ok := err.(net.Error); ok && netErr.Timeout() {
			return fmt.Errorf("email sending timed out: %w", err)
		}
		if strings.Contains(err.Error(), "authentication failed") {
			return fmt.Errorf("SMTP authentication failed: %w", err)
		}
		if strings.Contains(err.Error(), "connection refused") {
			return fmt.Errorf("connection to SMTP server refused: %w", err)
		}
		return fmt.Errorf("failed to send email: %w", err)
	}

	return nil
}
