package hirerpclogic

import (
	"context"
	"oa-server/app/hrcenter/model/hire"
	"sort"
	"strings"
	"sync"
	"time"

	"oa-server/app/hrcenter/hr_rpc/hr"
	"oa-server/app/hrcenter/hr_rpc/internal/svc"

	"github.com/zeromicro/go-zero/core/logx"
)

type GetJobPositionLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewGetJobPositionLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetJobPositionLogic {
	return &GetJobPositionLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *GetJobPositionLogic) GetJobPosition(in *hr.GetJobPositionReq) (*hr.GetJobPositionResp, error) {
	req := hire.FindAllJobPositionReq{
		StartTime:    in.StartTime,
		EndTime:      in.EndTime,
		DepartmentId: in.DepartmentId,
		PositionName: in.PositionName,
	}
	var positions []*hire.XPositionInfo
	var positionLangs []*hire.XPositionLanguageInfo
	var departmentLangMap map[int64]map[string]*hire.DepartmentLanguageInfo
	var deliverCounts map[uint64]int64

	positions, err := l.svcCtx.PositionInfoModel.QueryBasePositions(l.ctx, req)
	if err != nil {
		logx.Errorw("failed to get positionInfoList on GetJobPosition", logx.Field("error", err))
		return nil, ErrGetJobPositionFailed
	}

	// 收集所有需要查询的ID
	var positionIDs []uint64
	var departmentIDs []uint64
	for _, pos := range positions {
		positionIDs = append(positionIDs, uint64(pos.Id))
		departmentIDs = append(departmentIDs, uint64(pos.DepartmentId))
	}

	var wg sync.WaitGroup
	var err1, err2, err3 error
	wg.Add(3)

	go func() {
		defer wg.Done()
		positionLangs, err1 = l.svcCtx.PositionLanguageInfoModel.QueryPositionLanguages(l.ctx, positionIDs, "")
	}()

	go func() {
		defer wg.Done()
		_, departmentLangMap, err2 = l.svcCtx.DepartmentLanguageInfoModel.QueryDepartmentLanguages(l.ctx, departmentIDs, "")
	}()

	go func() {
		defer wg.Done()
		deliverCounts, err3 = l.svcCtx.DeliverRecordModel.QueryDeliverCounts(l.ctx, positionIDs)
	}()

	wg.Wait()

	if err1 != nil {
		logx.Errorw("failed to get positionLanguageInfo on GetJobPosition", logx.Field("error", err))
		return nil, ErrGetPositionLanguageFailed
	}
	if err2 != nil {
		logx.Errorw("failed to get departmentLanguageInfo on GetJobPosition", logx.Field("error", err))
		return nil, ErrGetDepartmentLanguageFailed
	}
	if err3 != nil {
		logx.Errorw("failed to get deliverCount on GetJobPosition", logx.Field("error", err))
		return nil, ErrGetDeliverCountFailed
	}

	return buildJobPositionResponse(positions, positionLangs, departmentLangMap, deliverCounts, in), nil
}

func buildJobPositionResponse(positions []*hire.XPositionInfo, positionLangs []*hire.XPositionLanguageInfo,
	departmentLangMap map[int64]map[string]*hire.DepartmentLanguageInfo, deliverCounts map[uint64]int64,
	req *hr.GetJobPositionReq) *hr.GetJobPositionResp {

	// 岗位多语言信息索引: positionID -> language -> info
	positionLangMap := make(map[uint64]map[string]*hire.XPositionLanguageInfo)
	for _, lang := range positionLangs {
		if _, ok := positionLangMap[uint64(lang.PositionId)]; !ok {
			positionLangMap[uint64(lang.PositionId)] = make(map[string]*hire.XPositionLanguageInfo)
		}
		positionLangMap[uint64(lang.PositionId)][lang.Language] = lang
	}

	// 构建返回结果
	resp := &hr.GetJobPositionResp{}
	for _, pos := range positions {
		// 获取岗位多语言信息
		var positionInfos []*hr.PositionInfo
		var IsContains bool

		if langs, ok := positionLangMap[uint64(pos.Id)]; ok {
			for langCode, langInfo := range langs {
				// 如果指定了岗位名称，是否包含
				if req.PositionName != "" && strings.Contains(langInfo.PositionName, req.PositionName) {
					IsContains = true
				}

				positionInfos = append(positionInfos, &hr.PositionInfo{
					Language:        langCode,
					PositionName:    langInfo.PositionName,
					OperatingDuty:   langInfo.OperatingDuty.String,
					JobRequirements: langInfo.JobRequirements.String,
				})
			}
		}

		// 如果没有匹配的岗位名称，跳过这个岗位
		if req.PositionName != "" && !IsContains {
			continue
		}

		// 获取部门名称(默认获取中文名称)
		departmentName := ""
		if deptLangs, ok := departmentLangMap[pos.DepartmentId]; ok {
			if deptLang, ok := deptLangs[Chinese]; ok {
				departmentName = deptLang.DepartmentName
			}
			// 如果默认语言找不到，尝试返回第一个找到的语言
			if departmentName == "" && len(deptLangs) > 0 {
				for _, v := range deptLangs {
					departmentName = v.DepartmentName
					break
				}
			}
		}

		// 获取投递记录数
		deliverCount := uint64(0)
		if count, ok := deliverCounts[uint64(pos.Id)]; ok {
			deliverCount = uint64(count)
		}

		// 构建最终岗位信息
		jobPos := &hr.JobPosition{
			PositionId:      uint64(pos.Id),
			Positions:       positionInfos,
			DepartmentId:    uint64(pos.DepartmentId),
			DepartmentName:  departmentName,
			Email:           pos.Email,
			Sort:            uint64(pos.Sort),
			DefaultLanguage: pos.Language,
			RecruitSwitch:   uint32(pos.RecruitSwitch),
			UrgentShortage:  uint32(pos.UrgentShortage),
			DeliverCount:    deliverCount,
			Creator:         pos.Creator,
			CreatedAt:       pos.CreatedAt.Format(time.DateTime),
		}

		resp.List = append(resp.List, jobPos)
	}

	sort.Slice(resp.List, func(i, j int) bool {
		return resp.List[i].Sort < resp.List[j].Sort
	})

	return resp
}
