package jobrpclogic

import (
	"context"
	"crypto/hmac"
	"crypto/sha256"
	"encoding/hex"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"oa-server/app/hrcenter/hr_rpc/internal/logic/common"
	"oa-server/app/hrcenter/model/hire"
	"oa-server/common/httputil"
	"oa-server/common/msgcenter"
	"strings"
	"time"

	"oa-server/app/hrcenter/hr_rpc/hr"
	"oa-server/app/hrcenter/hr_rpc/internal/svc"

	"github.com/zeromicro/go-zero/core/logx"
)

type DeliverResumeLogic struct {
	ctx        context.Context
	svcCtx     *svc.ServiceContext
	httpClient *httputil.Client
	logx.Logger
}

func NewDeliverResumeLogic(ctx context.Context, svcCtx *svc.ServiceContext) *DeliverResumeLogic {
	return &DeliverResumeLogic{
		ctx:        ctx,
		svcCtx:     svcCtx,
		httpClient: httputil.NewClient(30*time.Second, false),
		Logger:     logx.WithContext(ctx),
	}
}

type GetFileResponse struct {
	Data    map[string]string `json:"data"`
	Code    int               `json:"code"`
	Message string            `json:"message"`
}

type GetValidateResult struct {
	Status  string `json:"status"`
	Result  string `json:"result"`
	Reason  string `json:"reason"`
	Code    string `json:"code"`
	Message string `json:"msg"`
}

type ResumeData struct {
	Name string `json:"name"`
	Url  string `json:"url"`
}

func (l *DeliverResumeLogic) DeliverResume(in *hr.DeliverResumeReq) (*hr.DeliverResumeResp, error) {
	// 人机校验
	getValidateResult, err := l.GeeTestValidate(in.ValidateData)
	if err != nil {
		return nil, ErrDeliverResumeFailed
	}
	if getValidateResult.Status == ERRORSTATUS {
		return &hr.DeliverResumeResp{Success: true, Status: ValidateError}, nil
	}
	if getValidateResult.Status == SUCCESSSTATUS && getValidateResult.Result == FAILSTATUS {
		return &hr.DeliverResumeResp{Success: true, Status: ValidateFail}, nil
	}

	// 根据凭证获取简历链接
	getFileURL, err := url.JoinPath(l.svcCtx.Config.ImgOSS.FileURL, l.svcCtx.Config.ImgOSS.GetFilePath)
	if err != nil {
		return nil, ErrDeliverResumeFailed
	}

	var certificateList []string
	for _, resumeValue := range in.Resume {
		certificateList = append(certificateList, resumeValue.Certificate)
	}
	body := map[string]interface{}{
		"fileCertificates": certificateList,
	}
	bodyBytes, err := json.Marshal(body)
	if err != nil {
		return nil, ErrResumeJsonFailed
	}

	// 设置请求头
	headers := http.Header{}
	headers.Set("Content-Type", "application/json")

	getFileResp := &GetFileResponse{}
	_, _, _, err = l.httpClient.Post(context.Background(), getFileURL, headers, bodyBytes, getFileResp)
	if err != nil {
		return nil, ErrDeliverResumeFailed
	}

	var resumeList []ResumeData
	for _, resumeValue := range in.Resume {
		if value, exists := getFileResp.Data[resumeValue.Certificate]; exists {
			resumeList = append(resumeList, ResumeData{
				Name: resumeValue.Name,
				Url:  l.svcCtx.Config.ImgOSS.URL + "/" + value,
			})
		}
	}

	if len(resumeList) == 0 {
		return &hr.DeliverResumeResp{Success: true, Message: "请投递正确的简历！", Status: ValidateSuccess}, nil
	}

	resumeBytes, err := json.Marshal(resumeList)
	if err != nil {
		return nil, ErrResumeJsonFailed
	}

	// 获取这个邮箱是否投递过投递
	HasDelivered, err := l.svcCtx.DeliverRecordModel.HasDeliveredPosition(in.Email, in.PositionId)
	if err != nil {
		return nil, ErrHasDeliverResumeFailed
	}
	if HasDelivered {
		return &hr.DeliverResumeResp{Success: true, Message: "请勿重复投递！", Status: ValidateSuccess}, nil
	}

	// 判断本月是否已达到投递上限（3个）
	now := time.Now()
	firstDayOfMonth := time.Date(now.Year(), now.Month(), 1, 0, 0, 0, 0, now.Location())
	deliveryCount, err := l.svcCtx.DeliverRecordModel.DeliveryCount(in.Email, firstDayOfMonth)
	if err != nil {
		return nil, ErrDeliverCountFailed
	}
	if deliveryCount >= 3 {
		return &hr.DeliverResumeResp{Success: true, Message: "本月您已经投递过3个岗位，请下个月再进行投递!", Status: ValidateSuccess}, nil
	}

	exchangeExperience := NoExperience
	web3Experience := NoExperience
	if in.ExchangeName != "" {
		exchangeExperience = HasExperience
	}
	if in.Web3Name != "" {
		web3Experience = HasExperience
	}

	positionInfo := hire.XDeliverRecord{
		PositionId:         int64(in.PositionId),
		Name:               in.Name,
		Email:              in.Email,
		ExchangeExperience: exchangeExperience,
		Web3Experience:     web3Experience,
		ExchangeName:       in.ExchangeName,
		Web3Name:           in.Web3Name,
		Resume:             string(resumeBytes),
		CreatedAt:          now,
		UpdatedAt:          now,
	}

	insertResult, err := l.svcCtx.DeliverRecordModel.Insert(l.ctx, &positionInfo)
	if err != nil {
		return nil, ErrDeliverResumeFailed
	}

	// 异步发送lark
	id, _ := insertResult.LastInsertId()
	go l.SendCandidate(id, in)

	return &hr.DeliverResumeResp{Success: true, Status: ValidateSuccess}, nil
}

func (l *DeliverResumeLogic) SendCandidate(id int64, in *hr.DeliverResumeReq) error {
	ctx := context.Background()
	// 查询岗位
	positionInfo, err := l.svcCtx.PositionInfoModel.FindOne(ctx, int64(in.PositionId))
	if err != nil {
		return err
	}

	// 查询岗位多语言信息
	positionLangs, err := l.svcCtx.PositionLanguageInfoModel.QueryPositionLanguages(ctx, []uint64{in.PositionId}, "")
	if err != nil {
		return err
	}
	positionLangMap := make(map[string]*hire.XPositionLanguageInfo)
	for _, lang := range positionLangs {
		positionLangMap[lang.Language] = lang
	}

	var positionLangInfo *hire.XPositionLanguageInfo
	var found bool
	if positionLangInfo, found = positionLangMap[Chinese]; !found {
		if positionLangInfo, found = positionLangMap[English]; !found {
			if len(positionLangs) > 0 {
				positionLangInfo = positionLangs[0]
			}
		}
	}

	if positionInfo.Email == "" || id == 0 {
		positionInfo.Email = l.svcCtx.Config.Lark.TestEmail
	}
	// 发送lark通知
	common.AsyncSendLarkMsg(l.svcCtx.Config.Env, func() {
		url := l.svcCtx.Config.Lark.DeliverRecordUrl
		if l.svcCtx.Config.Env == TestEnv {
			positionInfo.Email = l.svcCtx.Config.Lark.TestEmail
			url = l.svcCtx.Config.Lark.TestDeliverRecordUrl
		}
		url = fmt.Sprintf("%v?position_id=%v", url, in.PositionId)
		msgData := map[string]any{
			"title":      "招聘通知",
			"content":    fmt.Sprintf("候选人 %v 投递了 %v,请及时查收!", in.Name, positionLangInfo.PositionName),
			"target_url": url,
		}
		msg, err := l.svcCtx.LarkService.BuildCardContent(l.svcCtx.Config.Lark.GenericTextTmpId, msgData)
		if err != nil {
			l.Logger.Errorw("failed to build lark card content", logx.Field("error", err))
			return
		}
		if err = l.svcCtx.LarkService.SendCardMessage(msgcenter.LarkReceiveIdTypeEmail, positionInfo.Email, msgcenter.LarkMsgTypeCard, msg); err != nil {
			l.Logger.Errorw("failed to send user lark msg", logx.Field("error", err))
			return
		}
	})
	return nil
}

func (l *DeliverResumeLogic) GeeTestValidate(data *hr.Validate) (*GetValidateResult, error) {
	clientType := strings.ToUpper(data.ClientType)
	captchaId := l.svcCtx.Config.Gt4.CaptchaData[clientType].CaptchaID
	captchaKey := l.svcCtx.Config.Gt4.CaptchaData[clientType].CaptchaKey

	// 生成签名
	signToken := hmac_encode(captchaKey, data.LotNumber)

	// 构造 x-www-form-urlencoded 格式的 body
	body := url.Values{}
	body.Set("captcha_id", captchaId)
	body.Set("lot_number", data.LotNumber)
	body.Set("captcha_output", data.CaptchaOutput)
	body.Set("pass_token", data.PassToken)
	body.Set("gen_time", data.GenTime)
	body.Set("sign_token", signToken)

	// 设置请求头
	headers := http.Header{}
	headers.Set("Content-Type", "application/x-www-form-urlencoded")

	// 发送请求
	url := l.svcCtx.Config.Gt4.Domain + "/validate"
	getValidateResult := &GetValidateResult{}
	_, _, _, err := l.httpClient.Post(context.Background(), url, headers, []byte(body.Encode()), getValidateResult)
	if err != nil {
		return nil, err
	}

	return getValidateResult, nil
}

func hmac_encode(key string, data string) string {
	mac := hmac.New(sha256.New, []byte(key))
	mac.Write([]byte(data))
	return hex.EncodeToString(mac.Sum(nil))
}
