package jobrpclogic

import (
	"context"
	"google.golang.org/protobuf/types/known/timestamppb"
	"oa-server/app/hrcenter/model/hire"
	"sort"
	"sync"

	"oa-server/app/hrcenter/hr_rpc/hr"
	"oa-server/app/hrcenter/hr_rpc/internal/svc"

	"github.com/zeromicro/go-zero/core/logx"
)

type GetJobLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewGetJobLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetJobLogic {
	return &GetJobLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *GetJobLogic) GetJob(in *hr.GetJobReq) (*hr.GetJobResp, error) {
	req := hire.FindAllJobPositionReq{
		RecruitSwitch: RecruitOpen,
	}
	var positions []*hire.XPositionInfo
	var positionLangs []*hire.XPositionLanguageInfo
	var departmentLangMap map[int64]map[string]*hire.DepartmentLanguageInfo

	positions, err := l.svcCtx.PositionInfoModel.QueryBasePositions(l.ctx, req)
	if err != nil {
		logx.Errorw("failed to get positionInfoList on GetJob", logx.Field("error", err))
		return nil, ErrGetJobPositionFailed
	}

	// 收集所有需要查询的ID
	var positionIDs []uint64
	var departmentIDs []uint64
	for _, pos := range positions {
		positionIDs = append(positionIDs, uint64(pos.Id))
		departmentIDs = append(departmentIDs, uint64(pos.DepartmentId))
	}

	var wg sync.WaitGroup
	var err1, err2 error
	wg.Add(2)

	go func() {
		defer wg.Done()
		positionLangs, err1 = l.svcCtx.PositionLanguageInfoModel.QueryPositionLanguages(l.ctx, positionIDs, "")
	}()

	go func() {
		defer wg.Done()
		_, departmentLangMap, err2 = l.svcCtx.DepartmentLanguageInfoModel.QueryDepartmentLanguages(l.ctx, departmentIDs, "")
	}()

	wg.Wait()

	if err1 != nil {
		logx.Errorw("failed to get positionLanguageInfo on GetJob", logx.Field("error", err))
		return nil, ErrGetPositionLanguageFailed
	}
	if err2 != nil {
		logx.Errorw("failed to get departmentLanguageInfo on GetJob", logx.Field("error", err))
		return nil, ErrGetDepartmentLanguageFailed
	}

	return buildJobPositionResponse(positions, positionLangs, departmentLangMap, in.Language), nil
}

func buildJobPositionResponse(positions []*hire.XPositionInfo, positionLangs []*hire.XPositionLanguageInfo,
	departmentLangMap map[int64]map[string]*hire.DepartmentLanguageInfo, language string) *hr.GetJobResp {

	// 岗位多语言信息索引: positionID -> language -> info
	positionLangMap := make(map[uint64]map[string]*hire.XPositionLanguageInfo)
	for _, lang := range positionLangs {
		if _, ok := positionLangMap[uint64(lang.PositionId)]; !ok {
			positionLangMap[uint64(lang.PositionId)] = make(map[string]*hire.XPositionLanguageInfo)
		}
		positionLangMap[uint64(lang.PositionId)][lang.Language] = lang
	}

	// 构建返回结果
	resp := &hr.GetJobResp{}
	for _, pos := range positions {
		// 获取岗位多语言信息
		if langs, ok := positionLangMap[uint64(pos.Id)]; ok {
			var langInfo *hire.XPositionLanguageInfo
			var found bool

			// 1. 优先使用请求的语言
			if langInfo, found = langs[language]; !found {
				// 2. 使用岗位的默认语言
				if langInfo, found = langs[pos.Language]; !found {
					// 3. 使用岗位的英文语言
					if langInfo, found = langs[English]; !found {
						// 4. 使用第一个找到的语言
						for _, v := range langs {
							langInfo = v
							found = true
							break
						}
					}
				}
			}

			if found {
				// 获取部门名称
				departmentName := ""
				if deptLangs, ok := departmentLangMap[pos.DepartmentId]; ok {
					// 1. 优先使用请求的语言
					if deptLang, ok := deptLangs[language]; ok {
						departmentName = deptLang.DepartmentName
					} else {
						//// 2. 使用岗位的默认语言
						//if deptLang, ok := deptLangs[pos.Language]; ok {
						//	departmentName = deptLang.DepartmentName
						//} else {
						// 3. 使用岗位的英文语言
						if deptLang, ok := deptLangs[English]; ok {
							departmentName = deptLang.DepartmentName
						} else {
							// 4. 使用第一个找到的语言
							for _, v := range deptLangs {
								departmentName = v.DepartmentName
								break
							}
						}
					}
				}

				job := &hr.Job{
					PositionId:      uint64(pos.Id),
					PositionName:    langInfo.PositionName,
					OperatingDuty:   langInfo.OperatingDuty.String,
					JobRequirements: langInfo.JobRequirements.String,
					DepartmentId:    uint64(pos.DepartmentId),
					DepartmentName:  departmentName,
					Email:           pos.Email,
					Sort:            uint64(pos.Sort),
					UrgentShortage:  uint32(pos.UrgentShortage),
					CreatedAt:       timestamppb.New(pos.CreatedAt),
				}
				resp.List = append(resp.List, job)
			}
		}
	}

	sort.Slice(resp.List, func(i, j int) bool {
		// 先比较紧急程度，2(紧急)排在前面
		if resp.List[i].UrgentShortage != resp.List[j].UrgentShortage {
			return resp.List[i].UrgentShortage > resp.List[j].UrgentShortage
		}
		// 紧急程度相同的情况下，再比较Sort
		if resp.List[i].Sort != resp.List[j].Sort {
			return resp.List[i].Sort < resp.List[j].Sort
		}
		// Sort也相同的情况下，按CreateTime降序排序
		return resp.List[i].CreatedAt.AsTime().After(resp.List[j].CreatedAt.AsTime())
	})
	return resp
}
