package hire

import (
	"context"
	"database/sql"
	"encoding/json"
	"fmt"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"oa-server/app/hrcenter/hr_rpc/hr"
	"strings"
	"time"
)

type FindAllJobPositionReq struct {
	StartTime     string
	EndTime       string
	DepartmentId  uint64
	PositionName  string
	RecruitSwitch uint64
}

type DepartmentLanguageInfo struct {
	Id             int64 // id
	Language       string
	DepartmentName string // 部门名称
	UpdatedAt      time.Time
	CreatedAt      time.Time
}

type UpdatePositionReq struct {
	Id              uint64
	DepartmentId    uint64
	Email           string
	Sort            uint64
	DefaultLanguage string
	RecruitSwitch   uint32
	UrgentShortage  uint32
}

type FindAllDeliverRecordReq struct {
	ID         uint64
	PositionID uint64
	Name       string
	Exchange   uint32
	Web3       uint32
}

type FindAllDeliverRecordResp struct {
	Id           int64     `db:"id"`            // id
	PositionId   int64     `db:"position_id"`   // 岗位信息id
	Name         string    `db:"name"`          // 姓名
	Email        string    `db:"email"`         // 邮箱
	ExchangeName string    `db:"exchange_name"` // 交易所名称
	Web3Name     string    `db:"web3_name"`     // web3名称
	Resume       string    `db:"resume"`        // 简历/作品集
	PositionName string    `db:"position_name"`
	CreatedAt    time.Time `db:"created_at"`
}

func (m *customXPositionInfoModel) TransCtx(ctx context.Context, fn func(ctx context.Context, s sqlx.Session) error) error {
	return m.conn.TransactCtx(ctx, func(ctx context.Context, s sqlx.Session) error {
		return fn(ctx, s)
	})
}

// QueryBasePositions 查询基础岗位信息
func (m *customXPositionInfoModel) QueryBasePositions(ctx context.Context, req FindAllJobPositionReq) ([]*XPositionInfo, error) {
	query := "SELECT * FROM x_position_info WHERE 1=1"
	var args []interface{}

	if req.DepartmentId > 0 {
		query += " AND DEPARTMENT_ID = ?"
		args = append(args, req.DepartmentId)
	}

	if req.RecruitSwitch != 0 {
		query += " AND recruit_switch = ?"
		args = append(args, req.RecruitSwitch)
	}

	if req.StartTime != "" {
		query += " AND created_at >= ?"
		args = append(args, req.StartTime)
	}

	if req.EndTime != "" {
		query += " AND created_at <= ?"
		args = append(args, req.EndTime)
	}

	query += " ORDER BY SORT ASC, created_at DESC"

	var positions []*XPositionInfo
	err := m.conn.QueryRows(&positions, query, args...)
	if err != nil {
		return nil, err
	}

	return positions, nil
}

func (m *customXPositionInfoModel) UpdateBasePositions(ctx context.Context, session sqlx.Session, req *UpdatePositionReq) error {
	updateQuery := `
        UPDATE x_position_info 
        SET 
            department_id = ?,
            email = ?,
            language = ?,
            sort = ?,
            recruit_switch = ?,
            urgent_shortage = ?,
            updated_at = NOW()
        WHERE id = ?
    `

	_, err := session.ExecCtx(ctx, updateQuery,
		req.DepartmentId,
		req.Email,
		req.DefaultLanguage,
		req.Sort,
		req.RecruitSwitch,
		req.UrgentShortage,
		req.Id,
	)
	if err != nil {
		return err
	}

	return nil
}

// BatchInsertWithSession 批量插入多语言岗位信息
func (m *customXPositionLanguageInfoModel) BatchInsertWithSession(conn sqlx.Session, data []*XPositionLanguageInfo) (sql.Result, error) {
	// 构建 VALUES 部分
	var valueStrings []string
	var valueArgs []interface{}
	placeholders := "(?, ?, ?, ?, ?, ?, ?)" // 7个占位符

	for _, item := range data {
		valueStrings = append(valueStrings, placeholders)
		valueArgs = append(valueArgs, item.PositionId)
		valueArgs = append(valueArgs, item.Language)
		valueArgs = append(valueArgs, item.PositionName)
		valueArgs = append(valueArgs, item.OperatingDuty.String)
		valueArgs = append(valueArgs, item.JobRequirements.String)
		valueArgs = append(valueArgs, item.CreatedAt)
		valueArgs = append(valueArgs, item.UpdatedAt)
	}

	query := fmt.Sprintf(`
        INSERT INTO %s 
        (position_id, language, position_name, operating_duty, job_requirements, created_at, updated_at) 
        VALUES %s`,
		m.table,
		strings.Join(valueStrings, ","))

	ret, err := conn.Exec(query, valueArgs...)
	return ret, err
}

func (m *customXPositionLanguageInfoModel) BatchUpdateWithSession(conn sqlx.Session, positionID uint64, data []*XPositionLanguageInfo) (sql.Result, error) {
	_, err := conn.Exec("DELETE FROM x_position_language_info WHERE position_id = ?", positionID)
	if err != nil {
		return nil, err
	}

	if len(data) == 0 {
		return nil, nil
	}
	return m.BatchInsertWithSession(conn, data)
}

func (m *customXPositionLanguageInfoModel) UpdateLanguageInfos(ctx context.Context, session sqlx.Session, positionID uint64, positionInfo []*hr.PositionInfo) error {
	// 删除旧的多语言信息
	_, err := session.ExecCtx(ctx, "DELETE FROM x_position_language_info WHERE position_id = ?", positionID)
	if err != nil {
		return err
	}

	// 如果没有提供新语言信息，直接返回
	if len(positionInfo) == 0 {
		return nil
	}

	// 批量插入新的多语言信息
	insertQuery := `
        INSERT INTO x_position_language_info (
            position_id, language, position_name, 
            operating_duty, job_requirements, 
            created_at, updated_at
        ) VALUES (?, ?, ?, ?, ?, NOW(), NOW())
    `

	for _, lang := range positionInfo {
		_, err = session.ExecCtx(ctx, insertQuery,
			positionID,
			lang.Language,
			lang.PositionName,
			lang.OperatingDuty,
			lang.JobRequirements,
		)
		if err != nil {
			return err
		}
	}

	return nil
}

// QueryPositionLanguages 查询多语言岗位信息
func (m *customXPositionLanguageInfoModel) QueryPositionLanguages(ctx context.Context, positionIDs []uint64, language string) ([]*XPositionLanguageInfo, error) {
	if len(positionIDs) == 0 {
		return nil, nil
	}

	var placeholders []string
	var args []interface{}
	for _, id := range positionIDs {
		placeholders = append(placeholders, "?")
		args = append(args, id)
	}

	query := fmt.Sprintf(`
        SELECT * 
        FROM x_position_language_info 
        WHERE POSITION_ID IN (%s)
    `, strings.Join(placeholders, ", "))

	// 添加语言条件
	if language != "" {
		query += " AND LANGUAGE = ?"
		args = append(args, language)
	}

	var langs []*XPositionLanguageInfo
	err := m.conn.QueryRows(&langs, query, args...)
	if err != nil {
		return nil, err
	}

	return langs, nil
}

// QueryDepartmentLanguages 查询部门多语言
func (m *customXDepartmentLanguageInfoModel) QueryDepartmentLanguages(ctx context.Context, departmentIDs []uint64, language string) ([]DepartmentLanguageInfo, map[int64]map[string]*DepartmentLanguageInfo, error) {
	var placeholders []string
	var args []interface{}
	for _, id := range departmentIDs {
		placeholders = append(placeholders, "?")
		args = append(args, id)
	}

	query := "SELECT * FROM x_department_language_info WHERE 1=1"

	if len(placeholders) != 0 {
		query += fmt.Sprintf(` AND id IN (%s)`, strings.Join(placeholders, ", "))
	}

	var langs []*XDepartmentLanguageInfo
	err := m.conn.QueryRows(&langs, query, args...)
	if err != nil {
		return nil, nil, err
	}

	var convertedList []DepartmentLanguageInfo
	departmentLangMap := make(map[int64]map[string]*DepartmentLanguageInfo)
	for _, item := range langs {
		var nameMap map[string]string
		if err := json.Unmarshal([]byte(item.DepartmentName), &nameMap); err != nil {
			continue
		}

		if departmentLangMap[item.Id] == nil {
			departmentLangMap[item.Id] = make(map[string]*DepartmentLanguageInfo)
		}

		// 遍历语言
		for lang, name := range nameMap {
			if language != "" && lang != language {
				continue
			}
			// 创建转换后的对象
			converted := DepartmentLanguageInfo{
				Id:             item.Id,
				Language:       lang,
				DepartmentName: name,
				UpdatedAt:      item.UpdatedAt,
				CreatedAt:      item.CreatedAt,
			}

			// 添加到列表
			convertedList = append(convertedList, converted)

			// 添加到map
			departmentLangMap[item.Id][lang] = &converted
		}
	}
	return convertedList, departmentLangMap, nil
}

// QueryDeliverCounts 查询投递记录数
func (m *customXDeliverRecordModel) QueryDeliverCounts(ctx context.Context, positionIDs []uint64) (map[uint64]int64, error) {
	if len(positionIDs) == 0 {
		return nil, nil
	}

	var placeholders []string
	var args []interface{}
	for _, id := range positionIDs {
		placeholders = append(placeholders, "?")
		args = append(args, id)
	}

	query := fmt.Sprintf(`
        SELECT POSITION_ID, count(*) as count 
        FROM x_deliver_record 
        WHERE POSITION_ID IN (%s) 
        GROUP BY POSITION_ID
    `, strings.Join(placeholders, ", "))

	type result struct {
		PositionID uint64 `db:"POSITION_ID"`
		Count      int64  `db:"count"`
	}

	var counts []result
	err := m.conn.QueryRows(&counts, query, args...)
	if err != nil {
		return nil, err
	}

	countMap := make(map[uint64]int64)
	for _, c := range counts {
		countMap[c.PositionID] = c.Count
	}

	return countMap, nil
}

func (m *customXDeliverRecordModel) QueryDeliverRecords(ctx context.Context, req FindAllDeliverRecordReq) ([]*FindAllDeliverRecordResp, error) {
	query := `
        SELECT 
            d.id,
            d.name,
            d.email,
            d.exchange_name,
            d.web3_name,
            d.resume,
            d.position_id,
            p.position_name,
            d.created_at
        FROM 
            x_deliver_record d
        LEFT JOIN 
            x_position_language_info p ON d.position_id = p.position_id AND p.language = 'zh-TW'
        WHERE 
            1=1
    `
	var args []interface{}

	// 添加查询条件
	if req.ID > 0 {
		query += " AND d.id = ?"
		args = append(args, req.ID)
	}

	if req.PositionID > 0 {
		query += " AND d.POSITION_ID = ?"
		args = append(args, req.PositionID)
	}

	if req.Name != "" {
		query += " AND d.NAME LIKE ?"
		args = append(args, "%"+req.Name+"%")
	}

	if req.Exchange > 0 {
		query += " AND d.EXCHANGE_EXPERIENCE = ?"
		args = append(args, req.Exchange)
	}

	if req.Web3 > 0 {
		query += " AND d.WEB3_EXPERIENCE = ?"
		args = append(args, req.Web3)
	}
	query += " ORDER BY d.created_at DESC"

	// 执行查询
	var deliverRecord []*FindAllDeliverRecordResp
	err := m.conn.QueryRows(&deliverRecord, query, args...)
	if err != nil {
		return nil, err
	}

	return deliverRecord, nil
}

func (m *customXDeliverRecordModel) HasDeliveredPosition(email string, positionID uint64) (bool, error) {
	var count int
	err := m.conn.QueryRow(&count, `
    SELECT COUNT(*) FROM x_deliver_record 
    WHERE email = ? AND position_id = ?`,
		email, positionID)
	if err != nil {
		return false, err
	}
	return count > 0, nil
}

func (m *customXDeliverRecordModel) DeliveryCount(email string, createAt time.Time) (int, error) {
	var count int
	err := m.conn.QueryRow(&count, `
        SELECT COUNT(*) FROM x_deliver_record 
        WHERE email = ? AND created_at >= ?`,
		email, createAt)
	if err != nil {
		return 0, err
	}
	return count, nil
}
