package logic

import (
	"context"
	"fmt"
	"oa-server/app/issuecenter/issuerpc/internal/svc"
	"oa-server/app/issuecenter/issuerpc/issuerpc"
	"oa-server/app/issuecenter/model"
	"oa-server/common/globalkey"
	"oa-server/common/msgcenter"
	"strconv"
	"time"

	larkim "github.com/larksuite/oapi-sdk-go/v3/service/im/v1"
	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type AddIssueLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewAddIssueLogic(ctx context.Context, svcCtx *svc.ServiceContext) *AddIssueLogic {
	return &AddIssueLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *AddIssueLogic) AddIssue(in *issuerpc.IssueReq) (*issuerpc.IssueResp, error) {
	l.Logger.Infof("添加issue信息")

	issueInfo := &model.XIssues{
		Title:           in.Title,
		Content:         in.Content,
		Severity:        in.Severity,
		Status:          in.Status,
		AssignedTo:      in.AssignedTo,
		Reporter:        in.Reporter,
		Tester:          in.Tester,
		AffectedVersion: in.AffectedVersion,
		FixVersion:      in.FixVersion,
		Attachments:     in.Attachments,
		ProcessPos:      1,
		Type:            in.Type,
		CreateAt:        time.Now().Unix(),
	}

	// 开始数据库事务
	var issueId int64
	err := l.svcCtx.Db.Transact(func(session sqlx.Session) error {

		// res, err := l.svcCtx.IssueModel.Insert(l.ctx, issueInfo)
		// if err != nil {
		// 	return err
		// }
		insertIssueQuery := `INSERT INTO x_issues (title, content, severity, status, assigned_to, reporter, tester, affected_version, fix_version, attachments, process_pos, type, create_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`
		res, err := session.ExecCtx(
			l.ctx, insertIssueQuery, issueInfo.Title, issueInfo.Content, issueInfo.Severity, issueInfo.Status, issueInfo.AssignedTo, issueInfo.Reporter, issueInfo.Tester, issueInfo.AffectedVersion, issueInfo.FixVersion, issueInfo.Attachments, issueInfo.ProcessPos, issueInfo.Type, issueInfo.CreateAt)
		if err != nil {
			return err
		}
		id, err := res.LastInsertId()
		if err != nil {
			return err
		}
		issueInfo.Id = id
		issueId = id

		// 根据issue类型创建流程
		if _, err := l.CreateIssueProcess(session, issueId, in); err != nil {
			l.Logger.Errorf("Error inserting issue process: %v", err)
			return err
		}
		// 创建好了更新下主表的数据
		_, err = session.Exec(
			`UPDATE x_issues SET status = ?, process_pos = ?, process_status = ?, process_approver = ?,
			process_approver_code = ? WHERE id = ?`,
			ApprovalStatusWaiting, 2, ApprovalStatusWaiting, in.Tester, TesterCode, issueId)
		if err != nil {
			return err
		}
		return nil
	})
	if err != nil {
		return nil, err
	}

	if err := l.SendNextProcessLark(l.ctx, issueInfo); err != nil {
		return nil, err
	}
	return &issuerpc.IssueResp{Id: issueId}, nil
}

func (l *AddIssueLogic) CreateIssueProcess(session sqlx.Session, issueId int64, issueReq *issuerpc.IssueReq) ([]*model.XIssuesProcess, error) {
	// 谁创建的->测试是谁->开发确认->开发修复->测试测试修复->完成
	var processes []*model.XIssuesProcess
	now := time.Now().Unix()
	switch issueReq.Type {
	case IssueDefaultType: // 普通问题类型
		// 初始化 issue 处理步骤
		processes = []*model.XIssuesProcess{
			// 1 创建者创建问题
			{
				IssueId:        issueId,
				Pos:            1,
				Approver:       issueReq.Reporter,
				ApproverCode:   Begin,
				Status:         ApprovalStatusConfirmed,
				ProcessHandler: "",
				Remark:         "",
				CreateAt:       now,
				UpdateAt:       0,
				Attachments:    "",
				Tag:            "",
			},

			// 2 测试
			{
				IssueId:        issueId,
				Pos:            2,
				Approver:       issueReq.Tester,
				ApproverCode:   TesterCode,
				Status:         ApprovalStatusWaiting,
				ProcessHandler: "",
				Remark:         "",
				CreateAt:       now,
				UpdateAt:       0,
				Attachments:    "",
				Tag:            "",
			},

			// 3 开发确认
			{
				IssueId:        issueId,
				Pos:            3,
				Approver:       "",
				ApproverCode:   DeveloperCode,
				Status:         ApprovalStatusPending,
				ProcessHandler: "",
				Remark:         "",
				CreateAt:       now,
				UpdateAt:       0,
				Attachments:    "",
				Tag:            "",
			},

			// 4 开发修复确认完成
			{
				IssueId:        issueId,
				Pos:            4,
				Approver:       "",
				ApproverCode:   DeveloperCode,
				Status:         ApprovalStatusPending,
				ProcessHandler: "",
				Remark:         "",
				CreateAt:       now,
				UpdateAt:       0,
				Attachments:    "",
				Tag:            "",
			},

			// 5 测试确认是否修复成功
			{
				IssueId:        issueId,
				Pos:            5,
				Approver:       "",
				ApproverCode:   TesterCode,
				Status:         ApprovalStatusPending,
				ProcessHandler: "",
				Remark:         "",
				CreateAt:       now,
				UpdateAt:       0,
				Attachments:    "",
				Tag:            "",
			},

			// 6 流程结束 已上线
			{
				IssueId:        issueId,
				Pos:            6,
				Approver:       "",
				ApproverCode:   TesterCode,
				Status:         ApprovalStatusPending,
				ProcessHandler: "",
				Remark:         "",
				CreateAt:       now,
				UpdateAt:       0,
				Attachments:    "",
				Tag:            "",
			},
		}
	default:
		return nil, fmt.Errorf("unsupported issue type %d", issueReq.Type)
	}

	query := `INSERT INTO x_issues_process (issue_id, pos, approver, approver_code, status, process_handler, remark, create_at, update_at, attachments, tag)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?,?)`

	for _, process := range processes {
		_, err := session.Exec(
			query, process.IssueId, process.Pos, process.Approver, process.ApproverCode,
			process.Status, process.ProcessHandler, process.Remark, process.CreateAt, process.UpdateAt, process.Attachments,
			process.Tag)
		if err != nil {
			l.Logger.Errorf("Error inserting issue process: %v", err)
			return nil, err
		}
	}
	return processes, nil
}

func (l *AddIssueLogic) SendNextProcessLark(ctx context.Context, issues *model.XIssues) error {
	vars := make(map[string]interface{})
	vars = map[string]interface{}{
		"title":      issues.Title,
		"desc":       issues.Content,
		"createTime": time.Now().Format("2006-01-02 15:04:05"),
		"nodeMsg":    fmt.Sprintf("<at email= %s ></at> 提交问题，待测试Leader <at email= %s ></at> 确认。", issues.Reporter, issues.Tester),
		"linkUrl":    "https://oa.lbk.world/work-order/details?id=" + strconv.Itoa(int(issues.Id)),
	}

	cardContent := &msgcenter.NewCardContent{
		Type: "template",
		Data: &msgcenter.NewCardContentData{
			TemplateID:   globalkey.LarkTemplateIDForIssue,
			TemplateVars: vars,
		},
	}

	content, err := cardContent.String()
	if err != nil {
		return err
	}

	sentEmails := make(map[string]bool)
	emails := []string{issues.Tester}

	for _, email := range emails {
		if _, ok := sentEmails[email]; !ok {
			sentEmails[email] = true
			l.svcCtx.LarkService.SendCardMessage(larkim.ReceiveIdTypeEmail, email, larkim.MsgTypeInteractive, content)
		}
	}
	return nil
}
