package logic

import (
	"context"
	"fmt"
	"oa-server/common/globalkey"
	"strconv"
	"time"

	larkim "github.com/larksuite/oapi-sdk-go/v3/service/im/v1"
	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"

	"oa-server/app/issuecenter/issuerpc/internal/svc"
	"oa-server/app/issuecenter/issuerpc/issuerpc"
	"oa-server/app/issuecenter/model"
	"oa-server/common/msgcenter"
)

type HandleProcessLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewHandleProcessLogic(ctx context.Context, svcCtx *svc.ServiceContext) *HandleProcessLogic {
	return &HandleProcessLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *HandleProcessLogic) HandleProcess(in *issuerpc.HandleProcessRequest) (*issuerpc.HandleProcessResponse, error) {
	if in.Id == 0 {
		return nil, fmt.Errorf("issue Id 不能为空")
	}

	if in.Status != ApprovalStatusConfirmed && in.Status != ApprovalStatusRejected {
		return nil, fmt.Errorf("issue状态参数 status 不正确")
	}

	var process model.XIssuesProcess
	// 根据 process id 获取对应流程信息
	err := l.svcCtx.Db.QueryRow(&process, "SELECT * FROM x_issues_process WHERE id = ? AND status = ?", in.Id, ApprovalStatusWaiting)
	if err != nil {
		return nil, fmt.Errorf("该 Issue process 已变更: %v", err)
	}

	var issue model.XIssues
	// 根据流程信息的 issueId 获取对应信息
	err = l.svcCtx.Db.QueryRow(&issue, "SELECT * FROM x_issues WHERE id = ?", process.IssueId)
	if err != nil {
		return nil, fmt.Errorf("获取Issue信息失败: %v", err)
	}

	if issue.Type != 1 {
		return nil, fmt.Errorf("不支持该 issue 类型")
	}

	// 开启事务处理 issue 流程
	currentTime := time.Now().Unix()
	err = l.svcCtx.Db.Transact(func(session sqlx.Session) error {
		// 更新Issue处理流程的处理状态
		res, err := session.Exec("UPDATE x_issues_process SET status = ?, remark = ?, process_handler = ?, attachments = ?,tag = ?, create_at = ? WHERE id = ? AND status = ?",
			in.Status, in.Remark, in.Handler, in.Attachments, in.Tag, currentTime, in.Id, ApprovalStatusWaiting)
		if err != nil {
			return fmt.Errorf("更新Issue流程状态异常: %v", err)
		}
		rowsAffected, err := res.RowsAffected()
		if err != nil {
			return fmt.Errorf("无法获取受影响的行数: %v", err)
		}
		if rowsAffected == 0 {
			return fmt.Errorf("更新Issue流程状态异常")
		}

		// 根据 AssignType 和 Status 执行不同的逻辑
		// 分配类型: 1 表示普通处理, 2 表示测试 leader 分配测试, 3 表示测试分配研发和产品
		switch in.AssignType {
		case 1:
			if in.Status == ApprovalStatusRejected {
				return handleFailure(session, process)
			} else if in.Status == ApprovalStatusConfirmed {
				return handleSuccess(session, process, in.Handler)
			}

		case 2:
			if in.Status == ApprovalStatusRejected {
				return handleFailure(session, process)
			} else if in.Status == ApprovalStatusConfirmed {
				err = assignTester(session, in, issue.Id)
				if err != nil {
					return err
				}
				return handleSuccess(session, process, in.Tester)
			}

		case 3:
			if in.Status == ApprovalStatusRejected {
				return handleFailure(session, process)
			} else if in.Status == ApprovalStatusConfirmed {
				err = assignDeveloperAndProductManager(session, in, issue.Id)
				if err != nil {
					return err
				}
				return handleSuccess(session, process, in.Developer)
			}
		default:
			if in.Status == ApprovalStatusRejected {
				return handleFailure(session, process)
			} else if in.Status == ApprovalStatusConfirmed {
				return handleSuccess(session, process, in.Handler)
			}
		}

		return nil
	})

	if err != nil {
		return nil, fmt.Errorf("处理Issue流程失败: %v", err)
	}

	var processRes model.XIssuesProcess
	// 根据 process id 获取对应流程信息
	err = l.svcCtx.Db.QueryRow(&processRes, "SELECT * FROM x_issues_process WHERE id = ?", in.Id)
	if err != nil {
		return nil, fmt.Errorf("获取 Issue process 失败: %v", err)
	}

	if err := l.SendNextProcessLark(l.ctx, &issue, &processRes, in); err != nil {
		return nil, err
	}
	return &issuerpc.HandleProcessResponse{
		Id: in.Id,
	}, nil
}

// 分配测试人员
func assignTester(session sqlx.Session, in *issuerpc.HandleProcessRequest, issueId int64) error {
	query := "UPDATE x_issues SET tester = ? WHERE id = ?"
	_, err := session.Exec(query, in.Tester, issueId)
	if err != nil {
		return fmt.Errorf("分配测试人员失败: %v", err)
	}
	return nil
}

// 分配研发或产品经理
func assignDeveloperAndProductManager(session sqlx.Session, in *issuerpc.HandleProcessRequest, issueId int64) error {
	_, err := session.Exec("UPDATE x_issues SET developer = ? WHERE id = ?", in.Developer, issueId)
	if err != nil {
		return fmt.Errorf("分配研发或产品经理失败: %v", err)
	}
	return nil
}

// 处理拒绝
func handleFailure(session sqlx.Session, process model.XIssuesProcess) error {
	//  更新关联 issue 为失败状态
	_, err := session.Exec("UPDATE x_issues SET status = ? WHERE id = ?", ApprovalStatusRejected, process.IssueId)
	if err != nil {
		return fmt.Errorf("更新Issue状态为失败失败: %w", err)
	}
	return nil
}

// 处理通过
func handleSuccess(session sqlx.Session, process model.XIssuesProcess, handle string) error {
	// 更新到下一处理步骤
	nextPos := process.Pos + 1

	_, err := session.Exec("UPDATE x_issues_process SET status = ? WHERE pos = ? AND issue_id = ?", ApprovalStatusWaiting, nextPos, process.IssueId)
	if err != nil {
		return fmt.Errorf("更新到下一处理步骤失败: %w", err)
	}

	// 获取下一个处理人的信息
	var nextProcess model.XIssuesProcess
	err = session.QueryRow(&nextProcess, "SELECT * FROM x_issues_process WHERE issue_id = ? AND pos = ?", process.IssueId, nextPos)
	if err != nil {
		return fmt.Errorf("获取下一个处理人信息异常: %w", err)
	}

	// 更新 issue 状态
	issue := buildIssueForStatusSuccess(nextProcess, process)
	_, err = session.Exec("UPDATE x_issues SET process_pos = ?, process_approver = ?, process_approver_code = ?, process_status = ?, process_id = ? WHERE id = ?",
		issue.ProcessPos, handle, issue.ProcessApproverCode, issue.ProcessStatus, issue.ProcessId, process.IssueId)
	if err != nil {
		return fmt.Errorf("更新Issue状态异常: %w", err)
	}

	return nil
}

// buildIssueForStatusSuccess 构建处理通过的 issue
func buildIssueForStatusSuccess(nextProcess model.XIssuesProcess, process model.XIssuesProcess) model.XIssues {
	return model.XIssues{
		ProcessPos:          nextProcess.Pos,
		ProcessApprover:     nextProcess.Approver,
		ProcessApproverCode: nextProcess.ApproverCode,
		ProcessStatus:       ApprovalStatusWaiting,
		ProcessId:           nextProcess.Id,
	}
}

func (l *HandleProcessLogic) SendNextProcessLark(ctx context.Context, issues *model.XIssues, process *model.XIssuesProcess, in *issuerpc.HandleProcessRequest) error {
	var nodeMsg string
	switch process.Status { // status:  2 审核同意  3 审核拒绝
	case ApprovalStatusConfirmed:
		if in.AssignType == 2 {
			nodeMsg = fmt.Sprintf("测试leader分配给 <at email= %s ></at> ，请尽快处理。", issues.Tester)
		} else if in.AssignType == 3 {
			nodeMsg = fmt.Sprintf("测试 <at email= %s ></at> 分配给 <at email= %s ></at>，请尽快处理。", issues.Tester, in.Developer)
		}
	case ApprovalStatusRejected:
		if in.AssignType == 2 {
			nodeMsg = fmt.Sprintf("<at email=%s></at> 已拒绝，请联系<font color='red'><at email=%s></at></font>确认。", in.Handler, in.Developer)
		} else if in.AssignType == 3 {
			nodeMsg = fmt.Sprintf("<at email=%s></at> 已拒绝，请联系<font color='red'><at email=%s></at> </font>确认。", in.Tester, in.Developer)
		}
	default:
		nodeMsg = "当前 Issue 状态已更新，请关注最新动态。"
	}

	vars := make(map[string]interface{})
	vars = map[string]interface{}{
		"title":      issues.Title,
		"desc":       issues.Content,
		"createTime": time.Now().Format("2006-01-02 15:04:05"),
		"nodeMsg":    nodeMsg,
		"linkUrl":    "https://oa.lbk.world/work-order/details?id=" + strconv.Itoa(int(issues.Id)),
	}

	cardContent := &msgcenter.NewCardContent{
		Type: "template",
		Data: &msgcenter.NewCardContentData{
			TemplateID:   globalkey.LarkTemplateIDForIssue,
			TemplateVars: vars,
		},
	}

	content, err := cardContent.String()
	if err != nil {
		return err
	}

	sentEmails := make(map[string]bool)
	emails := []string{issues.Tester, issues.Developer, issues.Reporter, in.Developer, in.Tester}

	for _, email := range emails {
		if _, ok := sentEmails[email]; !ok {
			sentEmails[email] = true
			l.svcCtx.LarkService.SendCardMessage(larkim.ReceiveIdTypeEmail, email, larkim.MsgTypeInteractive, content)
		}
	}
	return nil
}
