package logic

import (
	"context"
	"fmt"
	"strconv"
	"time"

	"oa-server/app/issuecenter/issuerpc/internal/svc"
	"oa-server/app/issuecenter/issuerpc/issuerpc"
	"oa-server/app/issuecenter/model"
	"oa-server/common/globalkey"
	"oa-server/common/msgcenter"

	larkim "github.com/larksuite/oapi-sdk-go/v3/service/im/v1"
	"github.com/zeromicro/go-zero/core/logx"
)

type TransferIssueLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewTransferIssueLogic(ctx context.Context, svcCtx *svc.ServiceContext) *TransferIssueLogic {
	return &TransferIssueLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *TransferIssueLogic) TransferIssue(in *issuerpc.TransferReq) (*issuerpc.UpdateRepairStatusResponse, error) {
	err := l.svcCtx.IssueModel.HandleProcess(l.ctx, in.Email, in.Approver, in.Attachments, in.Tag, in.Note, in.Id, in.Pos, in.HandleStatus, in.EstimatedTime)
	if err != nil {
		l.Logger.Errorf("Failed to fetch issues: %v", err)
		return nil, err
	}
	resp := issuerpc.UpdateRepairStatusResponse{
		Success: true,
	}
	go func() {
		issues, _ := l.svcCtx.IssueModel.FindOne(l.ctx, in.Id)
		issuesProcess, _ := l.svcCtx.IssueModel.ListIssueProcess(l.ctx, in.Id)
		if issues != nil {
			l.sendNextProcessLark(l.ctx, issues, issuesProcess)
		}
	}()

	return &resp, nil

}

func (l *TransferIssueLogic) sendNextProcessLark(ctx context.Context, issues *model.XIssues, process []*model.XIssuesProcess) error {
	emails := []string{issues.Reporter}
	testerEmail := ""
	developerEmail := ""
	var nodeMsg string
	switch issues.Status {
	case 1:
		switch issues.ProcessPos {
		case 2:
			testerEmail = process[1].Approver
			nodeMsg = fmt.Sprintf("问题由<at email= %s ></at>创建, 请测试<at email= %s ></at> 确认。", issues.Reporter, process[1].Approver)
		case 3:
			testerEmail = process[1].Approver
			developerEmail = process[2].Approver
			nodeMsg = fmt.Sprintf("问题已确认, 已分配开发<at email= %s ></at> 修复中。\n问题备注：%s", process[2].Approver, process[1].Remark)
		case 4:
			testerEmail = process[1].Approver
			developerEmail = process[2].Approver
			nodeMsg = fmt.Sprintf("开发<at email= %s ></at>已确认问题。正在修复中。\n问题备注：%s", process[3].Approver, process[2].Remark)
		case 5:
			testerEmail = process[1].Approver
			developerEmail = process[2].Approver
			nodeMsg = fmt.Sprintf("开发<at email= %s ></at>已修复问题问题。 请<at email= %s ></at> 测试验证。\n问题备注：%s", process[3].Approver, process[4].Approver, process[4].Remark)
		}
	case 2:
		testerEmail = process[1].Approver
		developerEmail = process[2].Approver
		nodeMsg = fmt.Sprintf("测试<at email= %s ></at>已测试通过。已上线", process[1].Approver)
	case 3:
		switch issues.ProcessPos {
		case 2:
			testerEmail = process[1].Approver
			nodeMsg = fmt.Sprintf("问题由<at email= %s ></at>创建, 测试<at email= %s ></at> 已驳回。", issues.Reporter, process[1].Approver)
		case 3:
			testerEmail = process[1].Approver
			developerEmail = process[2].Approver
			nodeMsg = fmt.Sprintf("测试<at email= %s ></at>问题已确认, 开发<at email= %s ></at> 驳回\n问题备注：%s", process[1].Approver, process[2].Approver, process[2].Remark)
		case 4:
			testerEmail = process[1].Approver
			developerEmail = process[2].Approver
			nodeMsg = fmt.Sprintf("开发<at email= %s ></at>驳回。\n问题备注：%s", process[3].Approver, process[3].Remark)
		case 5:
			testerEmail = process[1].Approver
			developerEmail = process[2].Approver
			nodeMsg = fmt.Sprintf("开发<at email= %s ></at>已修复问题问题。 测试<at email= %s ></at> 驳回。\n问题备注：%s", process[3].Approver, process[4].Approver, process[4].Remark)
		case 6:
			testerEmail = process[1].Approver
			developerEmail = process[2].Approver
			nodeMsg = fmt.Sprintf("上线失败 开发<at email= %s ></at>已修复问题问题。 测试<at email= %s ></at> 驳回。\n问题备注：%s", process[3].Approver, process[4].Approver, process[5].Remark)

		}
	}
	// vars := make(map[string]interface{})
	vars := map[string]interface{}{
		"title":      issues.Title,
		"desc":       issues.Content,
		"createTime": time.Now().Format("2006-01-02 15:04:05"),
		"env":        "test",
		"nodeMsg":    nodeMsg,
		"linkUrl":    "https://oa.lbk.world/work-order/details?id=" + strconv.Itoa(int(issues.Id)),
	}

	cardContent := &msgcenter.NewCardContent{
		Type: "template",
		Data: &msgcenter.NewCardContentData{
			TemplateID:   globalkey.LarkTemplateIDForIssue,
			TemplateVars: vars,
		},
	}

	content, err := cardContent.String()
	if err != nil {
		return err
	}
	if testerEmail != "" {
		emails = append(emails, testerEmail)
	}
	if developerEmail != "" {
		emails = append(emails, developerEmail)
	}
	for _, email := range emails {
		l.svcCtx.LarkService.SendCardMessage(larkim.ReceiveIdTypeEmail, email, larkim.MsgTypeInteractive, content)
	}
	return nil
}
