package logic

import (
	"context"
	"fmt"
	"oa-server/common/globalkey"
	"strconv"
	"time"

	larkim "github.com/larksuite/oapi-sdk-go/v3/service/im/v1"

	"github.com/pkg/errors"
	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"

	"oa-server/app/issuecenter/issuerpc/internal/svc"
	"oa-server/app/issuecenter/issuerpc/issuerpc"
	"oa-server/app/issuecenter/model"
	"oa-server/common/msgcenter"
)

type UpdateRepairStatusLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewUpdateRepairStatusLogic(ctx context.Context, svcCtx *svc.ServiceContext) *UpdateRepairStatusLogic {
	return &UpdateRepairStatusLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *UpdateRepairStatusLogic) UpdateRepairStatus(in *issuerpc.UpdateRepairStatusRequest) (*issuerpc.UpdateRepairStatusResponse, error) {
	var Issue model.XIssues
	var processId int64
	err := l.svcCtx.Db.Transact(func(session sqlx.Session) error {
		query := `SELECT * FROM x_issues WHERE id = ? LIMIT 1`
		err := session.QueryRow(&Issue, query, in.Id)
		if err != nil {
			return errors.Wrapf(err, "failed to get Issue with id %s", in.Id)
		}

		// 更新对应流程备注，根据状态确定 pos
		pos := 0
		status := ApprovalStatusPending
		processStatus := ApprovalStatusPending

		switch in.Status {
		case 1:
			pos = 3
			processStatus = ApprovalStatusFixing
			status = ApprovalStatusFixing // 修复中

			if in.IsRejected == Allow {
				updateProcessQuery := `UPDATE x_issues SET process_pos = ? WHERE id = ?`
				_, err := session.Exec(updateProcessQuery, 4, in.Id)
				if err != nil {
					return errors.Wrapf(err, "failed to update process field for id %s", in.Id)
				}
			} else if in.IsRejected == 2 {
				processStatus = ApprovalStatusRejected
				status = ApprovalStatusRejected
				updateProcessQuery := `UPDATE x_issues SET process_pos = ? WHERE id = ?`
				_, err := session.Exec(updateProcessQuery, 3, in.Id)
				if err != nil {
					return errors.Wrapf(err, "failed to update process field for id %s", in.Id)
				}
			}

		case 2:
			pos = 4
			processStatus = ApprovalStatusFixed // 已修复
			status = ApprovalStatusFixed

			if in.IsRejected == Allow {
				updateProcessQuery := `UPDATE x_issues SET process_pos = ? WHERE id = ?`
				_, err := session.Exec(updateProcessQuery, 5, in.Id)
				if err != nil {
					return errors.Wrapf(err, "failed to update process field for id %s", in.Id)
				}
			} else if in.IsRejected == 2 {
				processStatus = ApprovalStatusRejected
				status = ApprovalStatusRejected
				updateProcessQuery := `UPDATE x_issues SET process_pos = ? WHERE id = ?`
				_, err := session.Exec(updateProcessQuery, 4, in.Id)
				if err != nil {
					return errors.Wrapf(err, "failed to update process field for id %s", in.Id)
				}
			}

		case 3:
			pos = 5
			processStatus = ApprovalStatusValidated // 已测试
			status = ApprovalStatusValidated

			if in.IsRejected == Allow {
				updateProcessQuery := `UPDATE x_issues SET process_pos = ? WHERE id = ?`
				_, err := session.Exec(updateProcessQuery, 6, in.Id)
				if err != nil {
					return errors.Wrapf(err, "failed to update process field for id %s", in.Id)
				}
			} else if in.IsRejected == 2 {
				processStatus = ApprovalStatusRejected
				status = ApprovalStatusRejected
				updateProcessQuery := `UPDATE x_issues SET process_pos = ? WHERE id = ?`
				_, err := session.Exec(updateProcessQuery, 5, in.Id)
				if err != nil {
					return errors.Wrapf(err, "failed to update process field for id %s", in.Id)
				}
			}

		case 4:
			pos = 6
			processStatus = ApprovalStatusDeployed // 已上线
			status = ApprovalStatusDeployed

			if in.IsRejected == Allow {
				updateProcessQuery := `UPDATE x_issues SET process_pos = ? WHERE id = ?`
				_, err := session.Exec(updateProcessQuery, 7, in.Id)
				if err != nil {
					return errors.Wrapf(err, "failed to update process field for id %s", in.Id)
				}
			} else if in.IsRejected == 2 {
				processStatus = ApprovalStatusRejected
				status = ApprovalStatusRejected
				updateProcessQuery := `UPDATE x_issues SET process_pos = ? WHERE id = ?`
				_, err := session.Exec(updateProcessQuery, 6, in.Id)
				if err != nil {
					return errors.Wrapf(err, "failed to update process field for id %s", in.Id)
				}
			}

		default:
			return errors.New("unknown status")
		}

		currentTime := time.Now().Unix()
		// 更新对应流程备注和状态
		updateProcessQuery := `UPDATE x_issues_process SET remark = ?, status = ? , attachments = ?,tag = ?, process_handler = ?, create_at = ? WHERE issue_id = ? AND pos = ?`
		_, err = session.Exec(updateProcessQuery, in.Note, processStatus, in.Attachments, in.Tag, in.Handler, currentTime, Issue.Id, pos)
		if err != nil {
			return errors.Wrapf(err, "failed to update x_issues_process for issue id %d and pos %d", Issue.Id, pos)
		}

		// 获取更新后的 process_id
		getProcessIdQuery := `SELECT id FROM x_issues_process WHERE issue_id = ? AND pos = ?`
		err = session.QueryRow(&processId, getProcessIdQuery, Issue.Id, pos)
		if err != nil {
			return errors.Wrapf(err, "failed to get process_id for issue id %d and pos %d", Issue.Id, pos)
		}

		if in.IsRejected == 2 {
			// 处理拒绝逻辑
			err = handleRejection(session, Issue.Id, processId)
			if err != nil {
				return errors.Wrapf(err, "failed to handle failure for issue id %d", Issue.Id)
			}
		} else {
			var processApprover string

			switch in.Status {
			case 1:
				processApprover = Issue.Developer
			case 2, 3, 4:
				processApprover = Issue.Tester
			default:
				return errors.New("unknown status for process approver update")
			}

			// 更新 x_issues 中的 process_status, status 和 process_id
			updateStatusQuery := `UPDATE x_issues SET process_status = ?, status = ?, process_id = ?, process_approver = ?  WHERE id = ?`
			_, err = session.Exec(updateStatusQuery, processStatus, status, processId, processApprover, in.Id)
			if err != nil {
				return errors.Wrapf(err, "failed to update process_status, status and process_id for id %s", in.Id)
			}
		}

		if err != nil {
			return errors.Wrapf(err, "failed to update x_issues_process for id %s", in.Id)
		}

		return nil
	})

	if err != nil {
		return nil, fmt.Errorf("处理issue流程失败: %v", err)
	}

	var processRes model.XIssuesProcess
	// 根据 process id 获取对应流程信息
	err = l.svcCtx.Db.QueryRow(&processRes, "SELECT * FROM x_issues_process WHERE id = ?", processId)
	if err != nil {
		return nil, fmt.Errorf("获取 Issue process 失败: %v", err)
	}

	if err := l.SendNextProcessLark(l.ctx, &Issue, &processRes, in); err != nil {
		return nil, err
	}

	return &issuerpc.UpdateRepairStatusResponse{Success: true}, nil
}

func handleRejection(session sqlx.Session, issueId int64, issueProcessId int64) error {
	_, err := session.Exec("UPDATE x_issues SET status = ? WHERE id = ?", ApprovalStatusRejected, issueId)
	if err != nil {
		return fmt.Errorf("更新 Issue 状态为失败错误: %w", err)
	}

	_, err = session.Exec("UPDATE x_issues_process SET status = ? WHERE id = ?", ApprovalStatusRejected, issueId)
	if err != nil {
		return fmt.Errorf("更新Issue Process 状态为失败错误: %w", err)
	}
	return nil
}

const (
	Allow  = 1 // 允许
	Reject = 2 // 拒绝
)

func (l *UpdateRepairStatusLogic) SendNextProcessLark(ctx context.Context, issues *model.XIssues, process *model.XIssuesProcess, in *issuerpc.UpdateRepairStatusRequest) error {
	var nodeMsg string
	switch in.Status {
	/*
			status
			1 更新对应 issue process and pos = 3 数据(确认bug并修复中);  2 更新对应 issue process and pos = 4 数据(已修复);
		    3 更新对应 issue process and pos = 5 数据(已测试);  	      4 更新对应 issue process and pos = 6 数据(已上线);
	*/
	case 1:
		if in.IsRejected == Allow {
			nodeMsg = fmt.Sprintf("问题已确认, <at email= %s ></at> 修复中。\n问题备注：%s", issues.Developer, in.Note)
		} else {
			nodeMsg = fmt.Sprintf("<font color='red'><at email=%s></at> 已拒绝。请联系<at email=%s></at>确认。\n问题备注：%s</font>", issues.Developer, issues.Tester, in.Note)
		}
	case 2:
		if in.IsRejected == Allow {
			nodeMsg = fmt.Sprintf("<at email= %s ></at> 已修复。 请<at email= %s ></at> 测试验证。\n问题备注：%s", issues.Developer, issues.Tester, in.Note)
		} else {
			nodeMsg = fmt.Sprintf("<font color='red'> <at email=%s></at> 已拒绝。请联系<at email=%s></at>确认! \n问题备注：%s</font>", issues.Developer, issues.Tester, in.Note)
		}
	case 3:
		if in.IsRejected == Allow {
			nodeMsg = fmt.Sprintf("<at email= %s ></at>测试通过。准备上线 <at email= %s ></at>  。\n问题备注：%s", issues.Tester, issues.Developer, in.Note)
		} else {
			nodeMsg = fmt.Sprintf("<font color='red'> <at email=%s></at>已拒绝。请联系 <at email=%s></at> 确认。\n问题备注：%s</font>", issues.Tester, issues.Developer, in.Note)
		}
	case 4:
		if in.IsRejected == Allow {
			nodeMsg = fmt.Sprintf("已修复上线。测试：<at email= %s ></at> 研发：<at email= %s ></at> Reporter:<at email= %s ></at>。\n问题备注：%s", issues.Tester, issues.Developer, issues.Reporter, in.Note)
		} else {
			nodeMsg = fmt.Sprintf("<font color='red'>已拒绝。请联系测试：<at email=%s></at> 研发：<at email=%s></at>确认。\n问题备注：%s</font>", issues.Tester, issues.Developer, in.Note)
		}
	default:
		nodeMsg = "当前 Issue 状态已更新，请及时关注最新动态。"
	}

	vars := make(map[string]interface{})
	vars = map[string]interface{}{
		"title":      issues.Title,
		"desc":       issues.Content,
		"createTime": time.Now().Format("2006-01-02 15:04:05"),
		"env":        "test",
		"nodeMsg":    nodeMsg,
		"linkUrl":    "https://oa.lbk.world/work-order/details?id=" + strconv.Itoa(int(in.Id)),
	}

	cardContent := &msgcenter.NewCardContent{
		Type: "template",
		Data: &msgcenter.NewCardContentData{
			TemplateID:   globalkey.LarkTemplateIDForIssue,
			TemplateVars: vars,
		},
	}

	content, err := cardContent.String()
	if err != nil {
		return err
	}

	sentEmails := make(map[string]bool)
	emails := []string{issues.Tester, issues.Developer, issues.Reporter}

	for _, email := range emails {
		if _, ok := sentEmails[email]; !ok {
			sentEmails[email] = true
			l.svcCtx.LarkService.SendCardMessage(larkim.ReceiveIdTypeEmail, email, larkim.MsgTypeInteractive, content)
		}
	}
	return nil
}
