package model

import (
	"context"
	"database/sql"
	"fmt"
	"strings"
	"time"

	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

// 定义处理人的类型代码
const (
	Begin                = 0   // 流程开始
	DeveloperCode        = 1   // 开发
	DeveloperLeaderCode  = 2   // 开发组长
	TesterCode           = 3   // 测试
	TesterLeaderCode     = 4   // 测试组长
	OpsCode              = 5   // 运维
	OpsLeaderCode        = 6   // 运维组长
	CreatorCode          = 7   // issue创建人
	ProductCode          = 8   // 产品
	ProductLeaderCode    = 9   // 产品组长
	EndingCode           = 100 // 流程结束
	HandleProcessSuccess = 2   // 处理成功
	HandleProcessFail    = 3   // 处理失败
)

/*
需要再 xissuesmodel_gen.go 代码加入

	type (
		xIssuesModel interface {
	这里加入下面接口代码
	List(ctx context.Context, cond map[string]interface{}, limit, offset int) ([]*XIssues,int64, error)
	ListIssueProcess(ctx context.Context, issueId int64) ([]*XIssuesProcess, error)

防止每次生成代码覆盖这个 list 代码
*/
func (m *defaultXIssuesModel) List(ctx context.Context, cond map[string]interface{}, limit, offset int) ([]*XIssues, int64, error) {
	var issues []*XIssues
	var totalCount int64

	queryBuilder := strings.Builder{}
	queryBuilder.WriteString("SELECT * FROM " + m.table + " WHERE `is_deleted` = 0") // 加入逻辑删除的过滤条件

	countQueryBuilder := strings.Builder{}
	countQueryBuilder.WriteString("SELECT COUNT(*) FROM " + m.table + " WHERE `is_deleted` = 0") // 同样的逻辑删除过滤条件

	args := make([]interface{}, 0)

	if status, ok := cond["status"]; ok && status != 0 {
		queryBuilder.WriteString(" AND `status` = ?")
		countQueryBuilder.WriteString(" AND `status` = ?")
		args = append(args, status)
	}

	if title, ok := cond["title"]; ok && title != "" {
		queryBuilder.WriteString(" AND `title` LIKE ?")
		countQueryBuilder.WriteString(" AND `title` LIKE ?")
		args = append(args, "%"+title.(string)+"%")
	}

	if severity, ok := cond["severity"]; ok && severity != "" {
		queryBuilder.WriteString(" AND `severity` = ?")
		countQueryBuilder.WriteString(" AND `severity` = ?")
		args = append(args, severity)
	}

	queryBuilder.WriteString(" ORDER BY create_at DESC")

	queryBuilder.WriteString(fmt.Sprintf(" LIMIT %d OFFSET %d", limit, offset))
	query := queryBuilder.String()

	err := m.CachedConn.QueryRowsNoCacheCtx(ctx, &issues, query, args...)
	if err != nil {
		return nil, 0, err
	}

	countQuery := countQueryBuilder.String()
	err = m.CachedConn.QueryRowNoCacheCtx(ctx, &totalCount, countQuery, args...)
	if err != nil {
		return nil, 0, err
	}
	return issues, totalCount, nil
}

func (m *defaultXIssuesModel) ListIssueProcess(ctx context.Context, issueId int64) ([]*XIssuesProcess, error) {
	var processes []*XIssuesProcess
	query := `SELECT * FROM x_issues_process WHERE issue_id = ? ORDER BY pos ASC`
	err := m.CachedConn.QueryRowsNoCacheCtx(ctx, &processes, query, issueId)
	if err != nil {
		return nil, err
	}
	return processes, nil
}

// 我的提交
func (m *defaultXIssuesModel) MySubmitList(ctx context.Context, cond map[string]interface{}, limit, offset int, email string) ([]*XIssues, int64, error) {
	var issues []*XIssues
	var totalCount int64 = 0
	args := make([]interface{}, 0)
	queryBuilder := strings.Builder{}
	queryBuilder.WriteString("SELECT * FROM " + m.table + " WHERE reporter = ? ") // 加入逻辑删除的过滤条件
	args = append(args, email)

	countQueryBuilder := strings.Builder{}
	countQueryBuilder.WriteString("SELECT COUNT(*) FROM " + m.table + " WHERE reporter = ?") // 同样的逻辑删除过滤条件

	if status, ok := cond["status"]; ok && status != 0 {
		queryBuilder.WriteString(" AND `status` = ?")
		countQueryBuilder.WriteString(" AND `status` = ?")
		args = append(args, status)
	}

	if title, ok := cond["title"]; ok && title != "" {
		queryBuilder.WriteString(" AND `title` LIKE ?")
		countQueryBuilder.WriteString(" AND `title` LIKE ?")
		args = append(args, "%"+title.(string)+"%")
	}

	if severity, ok := cond["severity"]; ok && severity != "" {
		queryBuilder.WriteString(" AND `severity` = ?")
		countQueryBuilder.WriteString(" AND `severity` = ?")
		args = append(args, severity)
	}

	queryBuilder.WriteString(" ORDER BY create_at DESC")

	queryBuilder.WriteString(fmt.Sprintf(" LIMIT %d OFFSET %d", limit, offset))
	query := queryBuilder.String()

	countQuery := countQueryBuilder.String()
	err := m.CachedConn.QueryRowNoCacheCtx(ctx, &totalCount, countQuery, args...)
	if totalCount == 0 {
		return nil, 0, nil
	}
	if err != nil {
		return nil, 0, err
	}

	err = m.CachedConn.QueryRowsNoCacheCtx(ctx, &issues, query, args...)
	// if err == sqlx.ErrNotFound {
	// 	return nil, 0, nil
	// }
	if err == sql.ErrNoRows {
		return nil, 0, nil
	}
	if err != nil {
		return nil, 0, err
	}

	return issues, totalCount, nil
}

func (m *defaultXIssuesModel) MyHandleList(ctx context.Context, email string, limit, offset int) ([]XIssues, int64, error) {
	query := "select count(*) from x_issues where `process_approver` = ? and status = 1"
	var total int64 = 0
	err := m.CachedConn.QueryRowNoCacheCtx(ctx, &total, query, email)
	if total == 0 {
		return nil, 0, nil
	}
	if err != nil {
		return nil, 0, err
	}

	query = "select * from x_issues where `process_approver` = ? and status = 1 ORDER BY id DESC LIMIT ? OFFSET ?"

	var resp []XIssues
	err = m.CachedConn.QueryRowsNoCacheCtx(ctx, &resp, query, email, limit, offset)
	if err == sql.ErrNoRows {
		return []XIssues{}, 0, nil
	}
	if err != nil {
		return []XIssues{}, 0, err
	}

	return resp, total, nil
}

func (m *defaultXIssuesModel) MyDoneList(ctx context.Context, email string, limit, offset int) ([]XIssues, int64, error) {
	var issues []XIssues
	var totalCount int64 = 0

	countArgs := make([]interface{}, 0)
	countQueryBuilder := strings.Builder{}
	// countQueryBuilder.WriteString(
	// 	"SELECT COUNT(DISTINCT issue_id) FROM x_issues_process WHERE approver = ? AND `status` > 1"
	// )
	countQueryBuilder.WriteString(`
		SELECT COUNT(DISTINCT xip.issue_id) AS unique_issue_count
        FROM x_issues_process xip
        JOIN x_issues xi ON xip.issue_id = xi.id
        WHERE xip.approver = ? AND xip.status > 1 AND xi.status > 1;
	`)
	countArgs = append(countArgs, email)

	args := make([]interface{}, 0)
	queryBuilder := strings.Builder{}
	// queryBuilder.WriteString("SELECT x_issues.* FROM x_issues INNER JOIN (SELECT DISTINCT issue_id FROM x_issues_process WHERE process_handler = ? AND `status` > 1")
	queryBuilder.WriteString(`
    SELECT 
        x_issues.*
    	FROM x_issues 
    	INNER JOIN (
			SELECT DISTINCT issue_id 
			FROM x_issues_process 
			WHERE approver = ? 
			AND status > 1
			LIMIT ? OFFSET ?
    	) AS filtered_issues 
    	ON x_issues.id = filtered_issues.issue_id
		Where x_issues.status > 1
    	ORDER BY x_issues.id DESC
	`)
	args = append(args, email)
	// queryBuilder.WriteString(" LIMIT ? OFFSET ?) AS filtered_issues ON x_issues.id = filtered_issues.issue_id ORDER BY x_issues.id DESC")
	args = append(args, limit, offset)

	// 执行总数查询
	err := m.CachedConn.QueryRowNoCacheCtx(ctx, &totalCount, countQueryBuilder.String(), countArgs...)
	if err != nil {
		return nil, 0, err
	}

	// 执行分页查询
	err = m.CachedConn.QueryRowsNoCacheCtx(ctx, &issues, queryBuilder.String(), args...)
	if err != nil {
		return nil, 0, err
	}

	return issues, totalCount, nil
}

func (m *defaultXIssuesModel) GetApproverByPos(pos int64) int64 {
	switch pos {
	case 1:
		return 0
	case 2:
		return TesterCode
	case 3:
		return DeveloperCode
	case 4:
		return DeveloperCode
	case 5:
		return TesterCode
	case 6:
		return EndingCode
	default:
		return -1
	}
}

// 这里要要开启事务操作多个表
func (m *defaultXIssuesModel) HandleProcess(ctx context.Context, email, approver, attachments, tag, remark string, issue_id, pos, handleStatus, estimatedTime int64) error {
	// 目前就只有6步流程
	if pos > 6 {
		return fmt.Errorf("操作的节点异常")
	}

	now := time.Now().Unix()
	// 这是通过
	err := m.CachedConn.TransactCtx(ctx, func(ctx context.Context, session sqlx.Session) error {
		// 先获取下一个节点的信息
		var serr error
		if handleStatus == HandleProcessSuccess {
			var nextProcess XIssuesProcess
			// 还没到确认上线的节点
			if pos < 6 {
				serr = session.QueryRow(&nextProcess, "SELECT * FROM x_issues_process WHERE issue_id = ? AND pos = ?", issue_id, pos+1)
				if serr != nil {
					return fmt.Errorf("289获取下一个处理节点失败: %v", serr)
				}
				// 更新当前节点的状态
				res, serr := session.Exec(
					`UPDATE x_issues_process SET
					attachments = ?, tag = ?, status = ?, remark = ?, update_at = ? 
					WHERE issue_id = ? AND pos = ? AND approver = ?`,
					attachments, tag, HandleProcessSuccess, remark, now, issue_id, pos, email)
				if serr != nil {
					return fmt.Errorf("298更新当前处理节点失败: %v", serr)
				}
				if reRow, _ := res.RowsAffected(); reRow == 0 {
					return fmt.Errorf("298当前节点更新失败")
				}

				// 构建动态 SQL 语句
				var additionalField string
				switch pos {
				case 3:
					additionalField = "estimated_time = ?"
				}

				// 构建完整的 SQL 语句
				query := `
					UPDATE x_issues SET process_status = 1, update_at = ?, 
					process_approver = ?, process_pos = ?, process_approver_code = ?
				`

				if additionalField != "" {
					query += ", " + additionalField
				}
				query += " WHERE id = ?"

				// 执行 SQL 语句
				var args []interface{}
				args = append(args, now, approver, pos+1, nextProcess.ApproverCode)
				if additionalField != "" {
					args = append(args, estimatedTime)
				}
				args = append(args, issue_id)
				res, serr = session.Exec(query, args...)
				if serr != nil {
					return fmt.Errorf("354更新 issue 状态失败: %v", serr)
				}
				if reRow, _ := res.RowsAffected(); reRow == 0 {
					return fmt.Errorf("357issue 状态更新失败")
				}
				// 构建完整的 SQL 语句
				processQuery := `
					UPDATE x_issues_process SET status = 1, update_at = ?,
					approver = ?
				`
				if additionalField != "" {
					processQuery += ", " + additionalField
				}
				processQuery += " WHERE issue_id = ? AND pos = ?"

				// 执行 SQL 语句
				var processArgs []interface{}
				processArgs = append(processArgs, now, approver)
				if additionalField != "" {
					processArgs = append(processArgs, estimatedTime)
				}
				processArgs = append(processArgs, issue_id, pos+1)
				res, serr = session.Exec(processQuery, processArgs...)
				if serr != nil {
					return fmt.Errorf("378更新 issue 状态失败: %v", serr)
				}
				if reRow, _ := res.RowsAffected(); reRow == 0 {
					return fmt.Errorf("381issue 状态更新失败")
				}

				return serr
			} else {
				// 这个就是直接确认上线了
				res, serr := session.Exec(
					`UPDATE x_issues_process SET
					attachments = ?, tag = ?, status = ?, remark = ?, update_at = ? 
					WHERE issue_id = ? AND pos = ? AND approver = ?`,
					attachments, tag, HandleProcessSuccess, remark, now, issue_id, 6, email)
				if serr != nil {
					return fmt.Errorf("368更新当前处理节点失败: %v", serr)
				}
				if reRow, _ := res.RowsAffected(); reRow == 0 {
					return fmt.Errorf("371当前节点更新失败")
				}

				// 构建完整的 SQL 语句
				query := `
					UPDATE x_issues SET status = 2, process_status = 2, update_at = ?, 
					process_approver = ?, process_pos = ?
					WHERE id = ?`

				// 执行 SQL 语句
				var args []interface{}
				args = append(args, now, email, 6, issue_id)
				res, serr = session.Exec(query, args...)
				if serr != nil {
					return fmt.Errorf("385更新 issue 状态失败: %v", serr)
				}
				if reRow, _ := res.RowsAffected(); reRow == 0 {
					return fmt.Errorf("388issue 状态更新失败")
				}
				return serr
			}
		} else {
			// 更新当前节点的状态
			res, serr := session.Exec(
				"UPDATE x_issues_process SET update_at = ?, attachments = ?, tag = ?, status = ?, remark = ? WHERE issue_id = ? AND pos = ? AND approver = ?",
				now, attachments, tag, HandleProcessFail, remark, issue_id, pos, email)
			if serr != nil {
				return fmt.Errorf("394更新当前处理节点失败: %v", serr)
			}
			if reRow, _ := res.RowsAffected(); reRow == 0 {
				return fmt.Errorf("397当前节点更新失败")
			}
			// 更新 issue 的状态
			res, serr = session.Exec(
				"UPDATE x_issues SET process_status = 3, status = 3, update_at = ? WHERE id = ?",
				now, issue_id)
			if serr != nil {
				return fmt.Errorf("404更新 issue 状态失败: %v", serr)
			}
			if reRow, _ := res.RowsAffected(); reRow == 0 {
				return fmt.Errorf("407issue 状态更新失败")
			}
			return serr
		}
	})
	return err
}

var updateIssueField = []string{
	"title",
	"content",
	"severity",
	"status",
	"assigned_to",
	"reporter",
	"tester",
	"developer",
	"product_manager",
	"affected_version",
	"fix_version",
	"attachments",
	"type",
	// "estimated_time",
}

func (m *defaultXIssuesModel) UpdateIssue(ctx context.Context, data *XIssues) error {
	fieldsStr := strings.Join(updateIssueField, "=?,") + "=?"
	xIssuesIdKey := fmt.Sprintf("%s%v", cacheXIssuesIdPrefix, data.Id)
	_, err := m.ExecCtx(ctx, func(ctx context.Context, conn sqlx.SqlConn) (result sql.Result, err error) {
		query := fmt.Sprintf(
			"update %s set %s where `id` = ?", m.table, fieldsStr)
		return conn.ExecCtx(
			ctx, query,
			data.Title, data.Content, data.Severity, data.Status, data.AssignedTo,
			data.Reporter, data.Tester, data.Developer, data.ProductManager, data.AffectedVersion, data.FixVersion,
			data.Attachments, data.Type, data.Id)
	}, xIssuesIdKey)
	return err
}
