package logic

import (
	"context"
	"fmt"
	"oa-server/app/jobcenter/cmd/scheduler/internal/config"
	"oa-server/app/oacenter/model/okr"
	"time"

	"github.com/hibiken/asynq"
	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

// 修改OKR状态的定时任务
const (
	// 任务类型
	TaskTypeCheckOkrStatus = "check_okr_status"
	// 任务队列
	QueueOkrStatus = "okr_status"
	// 同步间隔
	OkrStatusSyncInterval = "0 0 * * *" // 每天0点执行
)

// OkrStatusScheduler Okr状态检测
type OkrStatusScheduler struct {
	ctx      context.Context
	config   *config.Config
	okrModel okr.XOkrPeriodModel
}

// NewOkrStatusScheduler 创建OKR状态检测调度器
func NewOkrStatusScheduler(ctx context.Context, config *config.Config) *OkrStatusScheduler {
	return &OkrStatusScheduler{
		ctx:      ctx,
		okrModel: okr.NewXOkrPeriodModel(sqlx.NewMysql(config.DB.DataSource)),
	}
}

// RegisterTasks 注册任务
func (l *OkrStatusScheduler) RegisterTasks(scheduler *asynq.Scheduler) error {
	task := asynq.NewTask(TaskTypeCheckOkrStatus, nil, asynq.Queue(QueueOkrStatus))
	if _, err := scheduler.Register(OkrStatusSyncInterval, task, asynq.Queue(QueueOkrStatus)); err != nil {
		return fmt.Errorf("注册检测OKR状态任务失败: %w", err)
	}

	return nil
}

// RegisterHandlers 注册任务处理器

func (l *OkrStatusScheduler) RegisterHandlers(mux *asynq.ServeMux) {
	mux.HandleFunc(TaskTypeCheckOkrStatus, l.Check)
}
func (l *OkrStatusScheduler) Check(ctx context.Context, t *asynq.Task) error {
	return nil
	// 获取当前时间
	now := time.Now()

	// 获取所有状态为2的OKR
	okrs, err := l.okrModel.GetOkrPeriodByStatus(2)
	if err != nil {
		logx.Errorf("获取所有状态为2的OKR失败: %v", err)
		return err
	}

	logx.Info("开始执行OKR状态检测任务")

	for _, okr := range okrs {
		// 计算开始日期+15天
		deadlineDate := okr.StartDate.AddDate(0, 0, 15)
		// 如果当前日期 >= 开始日期+15天，则需要更新状态
		if now.After(deadlineDate) || now.Equal(deadlineDate) {
			err := l.okrModel.SetStatusFromTo(okr.PeriodId, 2, 1)
			if err != nil {
				logx.Errorf("设置OKR状态失败: %v", err)
				return err
			}
		}
	}
	return nil
}
