package logic

import (
	"context"
	"fmt"
	"oa-server/app/jobcenter/cmd/scheduler/internal/config"
	"oa-server/common/tool"
	"time"

	"github.com/hibiken/asynq"
	"github.com/redis/go-redis/v9"
)

// SchedulerManager 调度器管理器
type SchedulerManager struct {
	ctx       context.Context
	rdb       *redis.Client
	scheduler *asynq.Scheduler
	srv       *asynq.Server
	mux       *asynq.ServeMux
}

// NewSchedulerManager 创建调度器管理器
func NewSchedulerManager(ctx context.Context, rdb *redis.Client, c config.Config) *SchedulerManager {
	// 创建 Redis 选项
	redisOpts := asynq.RedisClientOpt{
		Addr:     rdb.Options().Addr,
		Password: rdb.Options().Password,
		DB:       rdb.Options().DB,
	}

	// 获取当前时区
	loc := time.Local

	// 创建调度器
	scheduler := asynq.NewScheduler(redisOpts, &asynq.SchedulerOpts{
		Location: loc,
		LogLevel: asynq.InfoLevel,
	})

	// 创建服务器
	srv := asynq.NewServer(redisOpts, asynq.Config{
		Queues: map[string]int{
			QueueUserCenter: 1,
			QueueOkrStatus:  1,
			// 可以添加其他队列
		},
		LogLevel: asynq.InfoLevel,
	})

	// 创建 ServeMux
	mux := asynq.NewServeMux()

	return &SchedulerManager{
		ctx:       ctx,
		rdb:       rdb,
		scheduler: scheduler,
		srv:       srv,
		mux:       mux,
	}
}

// RegisterScheduler 注册调度器
func (m *SchedulerManager) RegisterScheduler(scheduler Scheduler) error {
	// 注册任务
	if err := scheduler.RegisterTasks(m.scheduler); err != nil {
		return fmt.Errorf("注册任务失败: %w", err)
	}

	// 注册处理器
	scheduler.RegisterHandlers(m.mux)

	return nil
}

// Start 启动调度器管理器
func (m *SchedulerManager) Start() error {
	// 启动调度器
	go func() {
		if err := m.scheduler.Run(); err != nil {
			fmt.Printf("调度器运行错误: %v\n", err)
		}
	}()

	// 启动服务器
	tool.GoSafe(func() {
		if err := m.srv.Run(m.mux); err != nil {
			fmt.Printf("服务器运行错误: %v\n", err)
		}
	})

	return nil
}

// Stop 停止调度器管理器
func (m *SchedulerManager) Stop() {
	// 停止调度器
	if m.scheduler != nil {
		m.scheduler.Shutdown()
	}

	// 停止服务器
	if m.srv != nil {
		m.srv.Shutdown()
	}

	// 等待一段时间让任务完成
	select {
	case <-time.After(5 * time.Second):
		fmt.Println("等待任务完成超时")
	case <-m.ctx.Done():
		fmt.Println("上下文已取消")
	}
}
