package logic

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"oa-server/app/oacenter/oa_rpc/client/calcrpc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"strconv"
	"time"

	"oa-server/app/jobcenter/cmd/scheduler/internal/config"
	"oa-server/app/jobcenter/model"
	"oa-server/common/httputil"
	"oa-server/common/tool"

	"github.com/hibiken/asynq"
	"github.com/redis/go-redis/v9"
	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/zrpc"
)

const (
	// 任务类型
	TaskTypeSyncUserCenter = "sync_usercenter"
	// 任务队列
	QueueUserCenter = "usercenter"
	// 同步间隔
	SyncInterval = "*/5 * * * *" // 每5分钟执行一次
)

// UserCenterScheduler 用户中心调度器
type UserCenterScheduler struct {
	ctx           context.Context
	userCenterSvc *model.UserCenterService
	httpClient    *httputil.Client
	config        *config.Config
	calcRpc       calcrpc.CalcRpc
}

// NewUserCenterScheduler 创建用户中心调度器
func NewUserCenterScheduler(ctx context.Context, rdb *redis.Client, config *config.Config) *UserCenterScheduler {
	return &UserCenterScheduler{
		ctx:           ctx,
		userCenterSvc: model.NewUserCenterService(rdb),
		httpClient:    httputil.NewClient(30*time.Second, false),
		config:        config,
		calcRpc:       calcrpc.NewCalcRpc(zrpc.MustNewClient(config.OACenterRpcConf)),
	}
}

// RegisterTasks 注册任务
func (l *UserCenterScheduler) RegisterTasks(scheduler *asynq.Scheduler) error {
	// 注册同步用户中心任务
	task := asynq.NewTask(TaskTypeSyncUserCenter, nil, asynq.Queue(QueueUserCenter))
	if _, err := scheduler.Register(SyncInterval, task, asynq.Queue(QueueUserCenter)); err != nil {
		return fmt.Errorf("注册同步用户中心任务失败: %w", err)
	}

	return nil
}

// RegisterHandlers 注册任务处理器
func (l *UserCenterScheduler) RegisterHandlers(mux *asynq.ServeMux) {
	// 注册同步用户中心任务处理器
	mux.HandleFunc(TaskTypeSyncUserCenter, l.HandleSyncUserCenter)
}

type UserCenterResp struct {
	Code    int         `json:"code"`
	Message string      `json:"message"`
	Data    interface{} `json:"data"`
}

// GetUserList 获取用户列表
func (l *UserCenterScheduler) GetUserList() error {
	// 获取用户列表
	userListURL, err := url.JoinPath(l.config.UserCenter.URL, l.config.UserCenter.UserListPath)
	if err != nil {
		return fmt.Errorf("构建用户列表URL失败: %w", err)
	}
	body := map[string]interface{}{
		"pageNum":  1,
		"pageSize": 999999999,
	}
	bodyBytes, err := json.Marshal(body)
	if err != nil {
		return fmt.Errorf("序列化请求体失败: %w", err)
	}

	// 设置请求头
	headers := http.Header{}
	headers.Set("Content-Type", "application/json")

	userResp := &model.UserListResp{}
	_, _, _, err = l.httpClient.Post(context.Background(), userListURL, headers, bodyBytes, userResp)
	if err != nil {
		return fmt.Errorf("获取用户列表失败: %w", err)
	}

	// 获取部门列表
	deptListURL, err := url.JoinPath(l.config.UserCenter.URL, l.config.UserCenter.DepartmentPath)
	if err != nil {
		return fmt.Errorf("构建部门列表URL失败: %w", err)
	}

	// 发送空对象作为请求体
	emptyBody := []byte("{}")
	deptResp := &model.DepartmentListResp{}
	_, _, _, err = l.httpClient.Post(l.ctx, deptListURL, headers, emptyBody, deptResp)
	if err != nil {
		return fmt.Errorf("获取部门列表失败: %w", err)
	}

	tool.GoSafe(func() {
		deptInfo := make([]*oa.Item, 0)
		for _, v := range deptResp.Data {
			deptInfo = append(deptInfo, &oa.Item{
				Key:   v.DepartmentKey,
				Value: strconv.Itoa(int(v.UserRole)),
			})
		}
		_, err := l.calcRpc.DepartmentModify(context.Background(), &calcrpc.DepartmentModifyReq{
			DepartmentInfo: deptInfo,
		})
		if err != nil {
			logx.Errorw("DepartmentModify failed", logx.Field("error", err))
		}
	})

	// 构建用户中心数据
	data := &model.UserCenterData{
		Users:       userResp.Data.List,
		Departments: deptResp.Data,
		UpdateTime:  time.Now(),
	}

	// 保存到Redis
	if err := l.userCenterSvc.SaveUserCenterData(l.ctx, data); err != nil {
		return fmt.Errorf("保存用户中心数据失败: %w", err)
	}

	logx.Infof("同步用户中心数据成功，用户数: %d, 部门数: %d", len(data.Users), len(data.Departments))
	return nil
}

// HandleSyncUserCenter 处理同步用户中心任务
func (l *UserCenterScheduler) HandleSyncUserCenter(ctx context.Context, t *asynq.Task) error {
	return l.GetUserList()
}
