package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"os/signal"
	"syscall"
	"time"

	"oa-server/app/jobcenter/cmd/scheduler/internal/config"
	"oa-server/app/jobcenter/cmd/scheduler/internal/logic"

	"github.com/redis/go-redis/v9"
	"github.com/zeromicro/go-zero/core/conf"
	"github.com/zeromicro/go-zero/core/logx"
)

func init() {
	// 设置时区为东八区（Asia/Shanghai）
	loc, err := time.LoadLocation("Asia/Shanghai")
	if err != nil {
		panic(fmt.Sprintf("加载时区失败: %v", err))
	}
	time.Local = loc
}

var configFile = flag.String("f", "etc/scheduler.yaml", "the config file")

func main() {
	flag.Parse()
	var c config.Config

	conf.MustLoad(*configFile, &c)

	logx.DisableStat()
	// log、prometheus、trace、metricsUrl.
	if err := c.SetUp(); err != nil {
		panic(err)
	}

	rdb := redis.NewClient(&redis.Options{
		Addr:     c.Redis.Host,
		Password: c.Redis.Pass,
		DB:       0,
	})

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	manager := logic.NewSchedulerManager(ctx, rdb, c)

	userCenterScheduler := logic.NewUserCenterScheduler(ctx, rdb, &c)
	// 先调用一次
	userCenterScheduler.GetUserList()
	// 注册用户中心调度器
	if err := manager.RegisterScheduler(userCenterScheduler); err != nil {
		logx.Errorf("注册用户中心调度器失败: %v", err)
	}
	okrStatusScheduler := logic.NewOkrStatusScheduler(ctx, &c)
	if err := manager.RegisterScheduler(okrStatusScheduler); err != nil {
		logx.Errorf("注册OKR状态检测调度器失败: %v", err)
	}

	// 启动调度器管理器
	if err := manager.Start(); err != nil {
		logx.Errorf("启动调度器管理器失败: %v", err)
	}

	defer manager.Stop()

	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM)

	// 等待信号
	sig := <-sigChan
	logx.Infof("收到信号: %v, 开始优雅关闭...", sig)

	cancel()

	select {
	case <-time.After(5 * time.Second):
		logx.Info("等待清理超时，强制退出")
	case <-ctx.Done():
		logx.Info("所有goroutine已清理完成")
	}
}
