package model

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"slices"
	"time"

	"github.com/redis/go-redis/v9"
)

const (
	// Redis key 前缀
	UserCenterKeyPrefix = "oa-server:usercenter:"
	// 用户中心数据过期时间
	UserCenterDataExpiration = 24 * time.Hour
	// 用户数据Key
	UserDataKey = UserCenterKeyPrefix + "users"
	// 部门数据Key
	DepartmentDataKey = UserCenterKeyPrefix + "departments"
	// 用户ID索引前缀
	UserIDPrefix = UserCenterKeyPrefix + "user:"
	// 部门Key索引前缀
	DepartmentKeyPrefix = UserCenterKeyPrefix + "department:"
)

// UserCenterService 用户中心服务接口
type UserCenterService struct {
	rdb *redis.Client
}

// NewUserCenterService 创建用户中心服务实例
func NewUserCenterService(rdb *redis.Client) *UserCenterService {
	return &UserCenterService{
		rdb: rdb,
	}
}

// SaveUserCenterData 保存用户中心数据
func (s *UserCenterService) SaveUserCenterData(ctx context.Context, data *UserCenterData) error {
	pipe := s.rdb.Pipeline()
	updateTime := time.Now()

	// 保存用户数据
	if err := s.saveUsers(ctx, pipe, data.Users, updateTime); err != nil {
		return err
	}

	// 保存部门数据
	if err := s.saveDepartments(ctx, pipe, data.Departments, updateTime); err != nil {
		return err
	}

	// 执行管道
	_, err := pipe.Exec(ctx)
	if err != nil {
		return fmt.Errorf("执行Redis管道失败: %w", err)
	}

	return nil
}

// saveUsers 保存用户数据
func (s *UserCenterService) saveUsers(ctx context.Context, pipe redis.Pipeliner, users []UserInfo, updateTime time.Time) error {
	// 序列化用户列表
	usersBytes, err := json.Marshal(users)
	if err != nil {
		return fmt.Errorf("序列化用户列表失败: %w", err)
	}

	// 保存用户列表
	pipe.Set(ctx, UserDataKey, usersBytes, UserCenterDataExpiration)

	// 保存单个用户索引
	for _, user := range users {
		userBytes, err := json.Marshal(user)
		if err != nil {
			return fmt.Errorf("序列化用户信息失败: %w", err)
		}

		// 使用username作为用户ID索引
		userKey := UserIDPrefix + user.Email
		pipe.Set(ctx, userKey, userBytes, UserCenterDataExpiration)
	}

	return nil
}

// saveDepartments 保存部门数据
func (s *UserCenterService) saveDepartments(ctx context.Context, pipe redis.Pipeliner, departments []DepartmentInfo, updateTime time.Time) error {
	// 序列化部门列表
	deptsBytes, err := json.Marshal(departments)
	if err != nil {
		return fmt.Errorf("序列化部门列表失败: %w", err)
	}

	// 保存部门列表
	pipe.Set(ctx, DepartmentDataKey, deptsBytes, UserCenterDataExpiration)

	// 保存单个部门索引
	for _, dept := range departments {
		deptBytes, err := json.Marshal(dept)
		if err != nil {
			return fmt.Errorf("序列化部门信息失败: %w", err)
		}

		// 使用department_key作为部门索引
		deptKey := DepartmentKeyPrefix + dept.DepartmentKey
		pipe.Set(ctx, deptKey, deptBytes, UserCenterDataExpiration)
	}

	return nil
}

// GetUserCenterData 获取用户中心数据
func (s *UserCenterService) GetUserCenterData(ctx context.Context) (*UserCenterData, error) {
	users, err := s.GetAllUsers(ctx)
	if err != nil {
		return nil, err
	}

	departments, err := s.GetAllDepartments(ctx)
	if err != nil {
		return nil, err
	}

	return &UserCenterData{
		Users:       users,
		Departments: departments,
		UpdateTime:  time.Now(),
	}, nil
}

// GetAllUsers 获取所有用户
func (s *UserCenterService) GetAllUsers(ctx context.Context) ([]UserInfo, error) {
	// 从Redis获取用户列表
	usersBytes, err := s.rdb.Get(ctx, UserDataKey).Bytes()
	if err != nil {
		return nil, fmt.Errorf("从Redis获取用户列表失败: %w", err)
	}

	// 反序列化用户列表
	var users []UserInfo
	if err := json.Unmarshal(usersBytes, &users); err != nil {
		return nil, fmt.Errorf("反序列化用户列表失败: %w", err)
	}

	return users, nil
}

// GetAllDepartments 获取所有部门
func (s *UserCenterService) GetAllDepartments(ctx context.Context) ([]DepartmentInfo, error) {
	// 从Redis获取部门列表
	deptsBytes, err := s.rdb.Get(ctx, DepartmentDataKey).Bytes()
	if err != nil {
		return nil, fmt.Errorf("从Redis获取部门列表失败: %w", err)
	}

	// 反序列化部门列表
	var departments []DepartmentInfo
	if err := json.Unmarshal(deptsBytes, &departments); err != nil {
		return nil, fmt.Errorf("反序列化部门列表失败: %w", err)
	}

	return departments, nil
}

func (s *UserCenterService) GetDepartmentMap(ctx context.Context) (map[string]DepartmentInfo, error) {
	departments, err := s.GetAllDepartments(ctx)
	if err != nil {
		return nil, err
	}
	departmentMap := make(map[string]DepartmentInfo)
	for _, dept := range departments {
		departmentMap[dept.DepartmentKey] = dept
	}
	return departmentMap, nil
}

// GetUserByEmail 根据email获取用户信息
func (s *UserCenterService) GetUserByEmail(ctx context.Context, employee string) (*UserInfo, error) {
	// 直接通过索引获取用户信息
	userKey := UserIDPrefix + employee
	userBytes, err := s.rdb.Get(ctx, userKey).Bytes()
	if errors.Is(err, redis.Nil) {
		return nil, fmt.Errorf("未找到工号为 %s 的用户", employee)
	} else if err != nil {
		return nil, fmt.Errorf("从Redis获取用户信息失败: %w", err)
	}

	// 反序列化用户信息
	var user UserInfo
	if err := json.Unmarshal(userBytes, &user); err != nil {
		return nil, fmt.Errorf("反序列化用户信息失败: %w", err)
	}

	return &user, nil
}

// GetDepartmentByKey 根据部门ID获取部门信息
func (s *UserCenterService) GetDepartmentByKey(ctx context.Context, departmentKey string) (*DepartmentInfo, error) {
	// 直接通过索引获取部门信息
	deptKey := DepartmentKeyPrefix + departmentKey
	deptBytes, err := s.rdb.Get(ctx, deptKey).Bytes()
	if errors.Is(err, redis.Nil) {
		return &DepartmentInfo{DepartmentName: "未知部门", DepartmentKey: "未知部门"}, nil
	} else if err != nil {
		return nil, fmt.Errorf("从Redis获取部门信息失败: %w", err)
	}

	// 反序列化部门信息
	var department DepartmentInfo
	if err := json.Unmarshal(deptBytes, &department); err != nil {
		return nil, fmt.Errorf("反序列化部门信息失败: %w", err)
	}

	return &department, nil
}

// GetParentDepartmentByKey 根据部门ID获取父部门信息
func (s *UserCenterService) GetParentDepartmentByKey(ctx context.Context, departmentKey string) (*DepartmentInfo, error) {
	departmentMap, err := s.GetDepartmentMap(ctx)
	if err != nil {
		return nil, err
	}
	department, ok := departmentMap[departmentKey]
	if !ok {
		return nil, fmt.Errorf("未找到部门 %s", departmentKey)
	}
	return s.GetDepartmentByKey(ctx, department.ParentDepartmentKey)
}

// GetUsersByDepartment 获取部门下的所有用户
func (s *UserCenterService) GetUsersByDepartment(ctx context.Context, departmentID string) ([]UserInfo, error) {
	users, err := s.GetAllUsers(ctx)
	if err != nil {
		return nil, err
	}

	var departmentUsers []UserInfo
	for _, user := range users {
		for _, deptKey := range user.DepartmentKeyList {
			if deptKey == departmentID {
				departmentUsers = append(departmentUsers, user)
				break
			}
		}
	}

	return departmentUsers, nil
}

// 用户中心的DepartmentKeyList并不一定包含用户的secondDepartment，所以想要查询某个二级部门在lark组织架构下的成员列表
// 则需要通过secondDepartmentList中取。
func (s *UserCenterService) GetUsersBySecondDepartment(ctx context.Context, departmentKey string) ([]UserInfo, error) {
	users, err := s.GetAllUsers(ctx)
	if err != nil {
		return nil, err
	}
	var departmentUsers []UserInfo
	for _, user := range users {
		for _, d := range user.SecondDepartmentList {
			if d.DepartmentKey == departmentKey {
				departmentUsers = append(departmentUsers, user)
				break
			}
		}
	}

	return departmentUsers, nil
}

// GetAllUsersByDepartment 获取当前部门及组(如果有)下的所有用户
// 这里获取的数据不准确，从用户中心直接获取
//func (s *UserCenterService) GetAllUsersByDepartment(ctx context.Context, departmentsID []string) ([]UserInfo, error) {
//	allDepartments, err := s.GetAllDepartments(ctx)
//	if err != nil {
//		return nil, err
//	}
//	deptsKeys := make([]string, 0)
//	for _, v := range allDepartments {
//		if slices.Contains(departmentsID, v.DepartmentKey) || slices.Contains(departmentsID, v.ParentDepartmentKey) {
//			deptsKeys = append(deptsKeys, v.DepartmentKey)
//		}
//	}
//	deptsKeys = append(deptsKeys, departmentsID...)
//	users, err := s.GetUsersByDepartments(ctx, deptsKeys)
//	if err != nil {
//		return nil, err
//	}
//	return users, nil
//}

func (s *UserCenterService) GetChildDepartment(ctx context.Context, departmentID string) ([]DepartmentInfo, error) {
	allDepartments, err := s.GetAllDepartments(ctx)
	if err != nil {
		return nil, fmt.Errorf("获取所有部门失败: %w", err)
	}

	// 查找指定部门
	for _, dept := range allDepartments {
		if dept.DepartmentKey == departmentID {
			// 找到部门，查找其所有子部门
			var childDepts []DepartmentInfo
			for _, childDept := range allDepartments {
				if childDept.ParentDepartmentKey == departmentID {
					childDepts = append(childDepts, childDept)
				}
			}
			return append([]DepartmentInfo{dept}, childDepts...), nil
		}
	}

	return nil, nil
}

// GetUsersByDepartments 获取部门下的所有用户
func (s *UserCenterService) GetUsersByDepartments(ctx context.Context, departmentKeys []string) ([]UserInfo, error) {
	users, err := s.GetAllUsers(ctx)
	if err != nil {
		return nil, err
	}

	var departmentUsers []UserInfo
	for _, user := range users {
		for _, dept := range user.DepartmentList {
			if slices.Contains(departmentKeys, dept.DepartmentKey) {
				departmentUsers = append(departmentUsers, user)
				break
			}
		}

	}

	return departmentUsers, nil
}

func (s *UserCenterService) GetLeaderDepartmentsMap(ctx context.Context) (map[string][]DepartmentInfo, error) {
	allDepartment, err := s.GetAllDepartments(ctx)
	if err != nil {
		return nil, err
	}
	ret := make(map[string][]DepartmentInfo)
	for _, dpt := range allDepartment {
		if dpt.Owner == "" {
			continue
		}
		ret[dpt.Owner] = append(ret[dpt.Owner], dpt)
	}
	return ret, nil
}
