package model

import (
	"context"
	"fmt"
	"github.com/zeromicro/go-zero/core/logx"
	"strings"
	"time"

	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type XReviewUsers struct {
	Id            uint64    `db:"id"`             // 主键id
	Level         string    `db:"level"`          // 故障定级
	IssueId       uint64    `db:"issue_id"`       // issue_id
	Scope         string    `db:"scope"`          // 影响范围
	Review        string    `db:"review"`         // 回顾
	RootCause     string    `db:"root_cause"`     // 故障根因
	Scheme        string    `db:"scheme"`         // 修复方案
	Improvement   string    `db:"improvement"`    // 改进方案
	Kpi           string    `db:"kpi"`            // 关联绩效
	Remark        string    `db:"remark"`         // 备注
	Owner         string    `db:"owner"`          // 创建人
	CreatedAt     time.Time `db:"created_at"`     // 创建时间
	UpdatedAt     time.Time `db:"updated_at"`     // 更新时间
	Responsible   string    `db:"responsible"`    // 责任人
	Handler       string    `db:"handler"`        // 处理人
	DepartmentKey string    `db:"department_key"` // 部门key
	Department    string    `db:"department"`     // 部门
	TeamKey       string    `db:"team_key"`       // 组key
	Team          string    `db:"team"`           // 组
	Users         string    `db:"users"`
}

// 更新的时候使用事务
func (m *customXReviewModel) UpdateAndUser(ctx context.Context, data *XReview, users string) error {
	err := m.conn.Transact(func(session sqlx.Session) error {
		// 更新review
		query := fmt.Sprintf("update %s set %s where `id` = ?", m.table, xReviewRowsWithPlaceHolder)
		_, err := session.Exec(
			query,
			data.Level,
			data.IssueId,
			data.Scope,
			data.Review,
			data.RootCause,
			data.Scheme,
			data.Improvement,
			data.Kpi,
			data.Remark,
			data.Owner,
			data.Responsible,
			data.Handler,
			data.DepartmentKey,
			data.Department,
			data.TeamKey,
			data.Team,
			data.Id,
		)
		if err != nil {
			return err
		}
		// 先将原有的用户删除
		_, err = session.Exec("delete from `x_review_user` where `rid` = ?", data.Id)
		if err != nil {
			return err
		}
		// 按,切割用户
		userList := strings.Split(users, ",")
		// 插入新的用户
		for _, user := range userList {
			_, err = session.Exec("insert into `x_review_user` (`rid`, `email`) values (?, ?)", data.Id, user)
			if err != nil {
				return err
			}
		}
		return nil
	})
	return err
}

func (m *customXReviewModel) InsertAndUser(ctx context.Context, data *XReview, users string) error {
	err := m.conn.Transact(func(session sqlx.Session) error {
		// 先插入review
		query := fmt.Sprintf("insert into %s (%s) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)", m.table, xReviewRowsExpectAutoSet)
		ret, err := session.Exec(
			query,
			data.Level,
			data.IssueId,
			data.Scope,
			data.Review,
			data.RootCause,
			data.Scheme,
			data.Improvement,
			data.Kpi,
			data.Remark,
			data.Owner,
			data.Responsible,
			data.Handler,
			data.DepartmentKey,
			data.Department,
			data.TeamKey,
			data.Team,
		)
		if err != nil {
			return err
		}
		id, err := ret.LastInsertId()
		if err != nil {
			return err
		}
		// 按,切割用户
		userList := strings.Split(users, ",")
		// 插入新的用户
		for _, user := range userList {
			_, err = session.Exec("insert into `x_review_user` (`rid`, `email`) values (?, ?)", id, user)
			if err != nil {
				return err
			}
		}
		return nil
	})
	return err
}

func (m *customXReviewModel) FindOneAndUser(ctx context.Context, id uint64) (*XReviewUsers, error) {
	// 分页获取数据
	var data XReviewUsers

	query := fmt.Sprintf("select %s, group_concat(`email`) as `users` from %s left join `x_review_user` on `x_review`.`id` = `x_review_user`.`rid` where `x_review`.`id` = ? group by `x_review`.`id` limit 1", xReviewRows, m.table)
	// query := fmt.Sprintf("select %s from %s left join `x_review_user` on `x_review`.`id` = `x_review_user`.`rid` where `x_review`.`id` = ? group by `x_review`.`id` limit 1", xReviewRows, m.table)
	err := m.conn.QueryRowCtx(ctx, &data, query, id)
	if err != nil {
		return nil, err
	}
	return &data, nil
}

func (m *customXReviewModel) GetList(ctx context.Context, page, pageSize uint64, levels, departmentKeys, teamKeys, responsible []string, startTime, endTime string) (int64, []*XReviewUsers, error) {
	var whereClause string
	var whereClauseParts []string
	var params []any

	if len(levels) > 0 {
		var placeHolders []string
		for _, level := range levels {
			placeHolders = append(placeHolders, "?")
			params = append(params, level)
		}
		whereClauseParts = append(
			whereClauseParts,
			fmt.Sprintf(" level in (%s) ", strings.Join(placeHolders, ",")))
	}
	if len(departmentKeys) > 0 {
		var likeConditions []string
		for _, key := range departmentKeys {
			// 构建 LIKE 条件，匹配 ,key, 、 key, 、 ,key 、 key 这几种情况
			//likeConditions = append(likeConditions, " ( department_key LIKE ? OR department_key LIKE ? OR department_key LIKE ? OR department_key = ? ) ")
			//params = append(params, "%,"+key+",%", key+",%", "%,"+key, key)
			likeConditions = append(likeConditions, " ( department_key like ? ) ")
			params = append(params, "%"+key+"%")
		}
		whereClauseParts = append(
			whereClauseParts,
			fmt.Sprintf(" ( %s ) ", strings.Join(likeConditions, " OR ")))
	}
	if len(teamKeys) > 0 {
		var likeConditions []string
		for _, key := range teamKeys {
			likeConditions = append(likeConditions, " ( team_key like ? ) ")
			params = append(params, "%"+key+"%")
		}
		whereClauseParts = append(
			whereClauseParts,
			fmt.Sprintf(" ( %s ) ", strings.Join(likeConditions, " OR ")))
	}
	fmt.Println(responsible, len(responsible))
	if len(responsible) > 0 {
		var likeConditions []string
		for _, key := range responsible {
			likeConditions = append(likeConditions, " ( responsible like ? ) ")
			params = append(params, "%"+key+"%")
		}
		whereClauseParts = append(
			whereClauseParts,
			fmt.Sprintf(" ( %s ) ", strings.Join(likeConditions, " OR ")))
	}

	if len(whereClauseParts) > 0 {
		whereClause = fmt.Sprintf(" where %s", strings.Join(whereClauseParts, " and "))
	}
	if startTime != "" {
		if whereClause != "" {
			whereClause = fmt.Sprintf("%s and created_at >= ?", whereClause)
			params = append(params, startTime)
		} else {
			whereClause = fmt.Sprintf(" where created_at >= ?")
			params = append(params, startTime)
		}
	}
	if endTime != "" {
		if whereClause != "" {
			whereClause = fmt.Sprintf("%s and created_at <=?", whereClause)
			params = append(params, endTime)
		} else {
			whereClause = fmt.Sprintf(" where created_at <=?")
			params = append(params, endTime)
		}
	}
	// 先查询总数
	var total int64
	query := fmt.Sprintf("select count(*) from %s %s", m.table, whereClause)
	err := m.conn.QueryRow(&total, query, params...)
	if err != nil {
		logx.Errorw("count review", logx.Field("error", err))
		return 0, nil, err
	}
	// 分页获取数据
	var list []*XReviewUsers
	params = append(params, (page-1)*pageSize)
	params = append(params, pageSize)
	query = fmt.Sprintf(`
select %s,
       group_concat(email)
           as users
from (select *
      from x_review
      %s )
      as x_review_t
         left join x_review_user
                   on x_review_t.id = x_review_user.rid
group by x_review_t.id
order by id desc
limit ?, ?
`, xReviewRows, whereClause)
	err = m.conn.QueryRowsCtx(ctx, &list, query, params...)
	if err != nil {
		logx.Errorw("list review", logx.Field("error", err))
		return 0, nil, err
	}
	return total, list, nil
}
