package efficiency_calc

import (
	"context"
	"fmt"
	"oa-server/app/oacenter/oa_rpc/oa"
	"strings"
)

type ListReq struct {
	Page             int
	PageSize         int
	Role             oa.Role
	DepartmentKey    []string
	ParentDepartment string
	Index            oa.CalcIndex
	ID               []int64
}

func (m *defaultXEfficiencyCalcModel) List(ctx context.Context, req *ListReq) (int64, []*EfficiencyCalcDTO, error) {
	if req.Page == 0 {
		req.Page = 1
	}
	if req.PageSize == 0 {
		req.PageSize = 10
	}
	offset := (req.Page - 1) * req.PageSize
	conditions := make([]string, 0)
	args := make([]any, 0)
	if req.Role != oa.Role_ROLE_UNSPECIFIED {
		conditions = append(conditions, "role = ?")
		args = append(args, req.Role)
	}

	if req.ParentDepartment != "" {
		conditions = append(conditions, "parent_department_key = ?")
		args = append(args, req.ParentDepartment)
	}
	if req.Index != oa.CalcIndex_UNKNOWN_CALC_INDEX {
		conditions = append(conditions, "calc_index = ?")
		args = append(args, req.Index)
	}
	if len(req.DepartmentKey) > 0 {
		placeholders := make([]string, len(req.DepartmentKey))
		for i := range placeholders {
			placeholders[i] = "?"
		}
		conditions = append(conditions, fmt.Sprintf("department_key IN (%s)", strings.Join(placeholders, ",")))
		for _, key := range req.DepartmentKey {
			args = append(args, key)
		}
	}
	if len(req.ID) > 0 {
		placeholders := make([]string, len(req.ID))
		for i := range placeholders {
			placeholders[i] = "?"
		}
		conditions = append(conditions, fmt.Sprintf("id IN (%s)", strings.Join(placeholders, ",")))
		for _, id := range req.ID {
			args = append(args, id)
		}
	}
	queryBuilder := strings.Builder{}
	queryBuilder.WriteString("SELECT * FROM ")
	queryBuilder.WriteString(m.table)

	if len(conditions) > 0 {
		queryBuilder.WriteString(" WHERE ")
		queryBuilder.WriteString(strings.Join(conditions, " AND "))
	}
	queryBuilder.WriteString(" ORDER BY created_at DESC")
	queryBuilder.WriteString(" LIMIT ? OFFSET ?")
	args = append(args, req.PageSize, offset)

	// 执行查询
	var list []*XEfficiencyCalc
	err := m.QueryRowsNoCache(&list, queryBuilder.String(), args...)
	if err != nil {
		return 0, nil, err
	}

	// 获取总数
	countQuery := fmt.Sprintf("SELECT COUNT(*) FROM %s", m.table)
	if len(conditions) > 0 {
		countQuery += " WHERE " + strings.Join(conditions, " AND ")
	}

	var total int64
	err = m.QueryRowNoCacheCtx(ctx, &total, countQuery, args[:len(args)-2]...)
	if err != nil {
		return 0, nil, err
	}
	var dtos []*EfficiencyCalcDTO
	for _, v := range list {
		dto, err := v.ToDTO()
		if err != nil {
			return 0, nil, err
		}
		dtos = append(dtos, dto)
	}
	return total, dtos, nil
}
