package efficiency_calc

import (
	"context"
	"fmt"
	"oa-server/app/oacenter/oa_rpc/oa"
	"strings"
	"time"
)

type ListHistoryReq struct {
	Page                  int
	PageSize              int
	CalcID                uint64
	DepartmentKey         string
	CalcIndex             oa.CalcIndex
	Role                  oa.Role
	EffectiveTimeLessThan time.Time
}

func (m *defaultXEfficiencyCalcHistoryModel) ListHistory(ctx context.Context, req *ListHistoryReq) (int64, []*EfficiencyCalcHistoryDTO, error) {
	if req.Page == 0 {
		req.Page = 1
	}
	if req.PageSize == 0 {
		req.PageSize = 10
	}
	offset := (req.Page - 1) * req.PageSize
	conditions := make([]string, 0)
	args := make([]any, 0)
	if req.CalcID != 0 {
		conditions = append(conditions, "calc_id = ?")
		args = append(args, req.CalcID)
	}
	if req.DepartmentKey != "" {
		conditions = append(conditions, "department_key = ?")
		args = append(args, req.DepartmentKey)
	}
	if req.Role != oa.Role_ROLE_UNSPECIFIED {
		conditions = append(conditions, "role = ?")
		args = append(args, req.Role)
	}

	if req.CalcIndex != oa.CalcIndex_UNKNOWN_CALC_INDEX {
		conditions = append(conditions, "calc_index = ?")
		args = append(args, req.CalcIndex)
	}

	if !req.EffectiveTimeLessThan.IsZero() {
		conditions = append(conditions, "effective_time <= ?")
		args = append(args, req.EffectiveTimeLessThan)
	}

	queryBuilder := strings.Builder{}
	queryBuilder.WriteString("SELECT * FROM ")
	queryBuilder.WriteString(m.table)

	if len(conditions) > 0 {
		queryBuilder.WriteString(" WHERE ")
		queryBuilder.WriteString(strings.Join(conditions, " AND "))
	}
	queryBuilder.WriteString(" ORDER BY created_at DESC")
	queryBuilder.WriteString(" LIMIT ? OFFSET ?")

	queryArgs := make([]any, len(args))
	copy(queryArgs, args)
	queryArgs = append(queryArgs, req.PageSize, offset)

	var list []*XEfficiencyCalcHistory
	err := m.QueryRowsNoCache(&list, queryBuilder.String(), queryArgs...)
	if err != nil {
		return 0, nil, err
	}

	// 获取总数
	countQuery := fmt.Sprintf("SELECT COUNT(*) FROM %s", m.table)

	if len(conditions) > 0 {
		countQuery += " WHERE " + strings.Join(conditions, " AND ")
	}

	var total int64
	err = m.QueryRowNoCacheCtx(ctx, &total, countQuery, args...)
	if err != nil {
		return 0, nil, err
	}

	var dtos []*EfficiencyCalcHistoryDTO
	for _, v := range list {
		dto, err := v.ToDTO()
		if err != nil {
			return 0, nil, err
		}
		dtos = append(dtos, dto)
	}
	return total, dtos, nil
}
