package issue

import (
	"context"
	"fmt"
	"oa-server/app/oacenter/oa_rpc/oa"
	"strings"
)

type DeleteReq struct {
	PeriodType int
	Employee   string
	DateFromTo string
	Department string
	Role       oa.Role
}

type ListReq struct {
	Page       int
	PageSize   int
	PeriodType int
	Employee   string
	Department string
	DateFromTo []string
	SortField  *oa.SortField
	Role       oa.Role
}

func (m *defaultXIssueMetricsModel) List(ctx context.Context, req *ListReq) (int64, []*IssueMetricsDTO, error) {
	if req.Page == 0 {
		req.Page = 1
	}
	if req.PageSize == 0 {
		req.PageSize = 10
	}

	// 构建查询条件
	var conditions []string
	var args []any

	if req.PeriodType != 0 {
		conditions = append(conditions, "period_type = ?")
		args = append(args, req.PeriodType)
	}

	if req.Employee != "" {
		conditions = append(conditions, "employee = ?")
		args = append(args, req.Employee)
	}

	if req.Department != "" {
		conditions = append(conditions, "department = ?")
		args = append(args, req.Department)
	}

	if len(req.DateFromTo) > 0 {
		placeholders := make([]string, len(req.DateFromTo))
		for i := range req.DateFromTo {
			placeholders[i] = "?"
			args = append(args, req.DateFromTo[i])
		}
		conditions = append(conditions, fmt.Sprintf("date_from_to IN (%s)", strings.Join(placeholders, ",")))
	}

	if req.Role != oa.Role_ROLE_UNSPECIFIED {
		conditions = append(conditions, "role = ?")
		args = append(args, req.Role)
	}

	// 构建基础查询
	queryBuilder := strings.Builder{}
	queryBuilder.WriteString("SELECT * FROM ")
	queryBuilder.WriteString(m.table)

	// 添加WHERE条件
	if len(conditions) > 0 {
		queryBuilder.WriteString(" WHERE ")
		queryBuilder.WriteString(strings.Join(conditions, " AND "))
	}

	// 添加排序条件
	if req.SortField != nil {
		queryBuilder.WriteString(fmt.Sprintf(" ORDER BY %s %s", req.SortField.Field, req.SortField.Order))
	} else {
		queryBuilder.WriteString(" ORDER BY score DESC")
	}

	queryBuilder.WriteString(" LIMIT ? OFFSET ?")
	args = append(args, req.PageSize, (req.Page-1)*req.PageSize)

	// 执行查询
	var list []*XIssueMetrics
	err := m.QueryRowsNoCache(&list, queryBuilder.String(), args...)
	if err != nil {
		return 0, nil, err
	}

	// 获取总数
	countQuery := fmt.Sprintf("SELECT COUNT(*) FROM %s", m.table)
	if len(conditions) > 0 {
		countQuery += " WHERE " + strings.Join(conditions, " AND ")
	}

	var total int64
	err = m.QueryRowNoCacheCtx(ctx, &total, countQuery, args[:len(args)-2]...)
	if err != nil {
		return 0, nil, err
	}
	var dtos []*IssueMetricsDTO
	for _, v := range list {
		dto, err := v.ToDTO()
		if err != nil {
			return 0, nil, err
		}
		var infos []*oa.Infos

		for _, info := range dto.Infos {
			infos = append(infos, &oa.Infos{
				Type:     WorkItemTypeMap[info.Type],
				Reason:   WorkItemReasonMap[info.Reason],
				Priority: WorkItemPriorityMap[info.Priority],
				Count:    info.Count,
			})
		}
		dto.Infos = infos
		dtos = append(dtos, dto)
	}

	return total, dtos, nil
}

func (m *defaultXIssueMetricsModel) DeleteByFiled(ctx context.Context, req DeleteReq) error {
	_, list, err := m.List(ctx, &ListReq{
		PeriodType: req.PeriodType,
		Employee:   req.Employee,
		Department: req.Department,
		DateFromTo: []string{req.DateFromTo},
		Role:       req.Role,
	})
	if err != nil {
		return err
	}
	if len(list) == 0 {
		return nil
	}

	return m.Delete(ctx, list[0].Id)

}
