package monitor_metrics

import (
	"context"
	"database/sql"
	"fmt"
	"strings"
	"time"

	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type ListReq struct {
	Page          int64    // 页码
	PageSize      int64    // 每页大小
	PeriodType    int64    // 周期类型
	Employee      string   // 员工
	DepartmentKey string   // 部门key
	DateFromTo    []string // 日期范围
	OrderBy       string   // 排序字段
}

func (m *defaultXMonitorMetricsModel) List(ctx context.Context, req *ListReq) (int64, []*XMonitorMetrics, error) {
	if req.Page == 0 {
		req.Page = 1
	}
	if req.PageSize == 0 {
		req.PageSize = 999999
	}
	var conditions []string
	var args []interface{}

	if req.PeriodType > 0 {
		conditions = append(conditions, "period_type = ?")
		args = append(args, req.PeriodType)
	}

	if req.Employee != "" {
		conditions = append(conditions, "employee = ?")
		args = append(args, req.Employee)
	}

	if req.DepartmentKey != "" {
		conditions = append(conditions, "department_key = ?")
		args = append(args, req.DepartmentKey)
	}

	if len(req.DateFromTo) > 0 {
		placeholders := make([]string, len(req.DateFromTo))
		for i := range req.DateFromTo {
			placeholders[i] = "?"
			args = append(args, req.DateFromTo[i])
		}
		conditions = append(conditions, fmt.Sprintf("date_from_to IN (%s)", strings.Join(placeholders, ",")))
	}

	conditions = append(conditions, "deleted_at IS NULL")

	var whereClause string
	if len(conditions) > 0 {
		whereClause = "WHERE " + strings.Join(conditions, " AND ")
	}

	orderBy := "score DESC"
	if req.OrderBy != "" {
		orderBy = req.OrderBy
	}

	offset := (req.Page - 1) * req.PageSize
	limit := req.PageSize

	// 查询总数
	countQuery := fmt.Sprintf("SELECT COUNT(*) FROM %s %s", m.table, whereClause)

	var total int64
	if err := m.CachedConn.QueryRowNoCacheCtx(ctx, &total, countQuery, args...); err != nil {
		logx.Errorf("Count error: %v", err)
		return 0, nil, err
	}

	// 如果没有记录，直接返回空结果
	if total == 0 {
		return 0, []*XMonitorMetrics{}, nil
	}

	// 构建查询语句
	query := fmt.Sprintf("SELECT %s FROM %s %s ORDER BY %s LIMIT ? OFFSET ?",
		xMonitorMetricsRows,
		m.table,
		whereClause,
		orderBy,
	)

	// 添加分页参数
	listArgs := make([]interface{}, len(args))
	copy(listArgs, args)
	listArgs = append(listArgs, limit, offset)

	// 查询数据
	var result []*XMonitorMetrics
	err := m.CachedConn.QueryRowsNoCacheCtx(ctx, &result, query, listArgs...)
	if err != nil {
		logx.Errorf("List query error: %v", err)
		return 0, nil, fmt.Errorf("查询监控指标失败: %v", err)
	}

	return total, result, nil
}

type DeleteReq struct {
	PeriodType int64  // 周期类型
	DateFromTo string // 日期范围
	Employee   string // 员工
}

func (m *defaultXMonitorMetricsModel) SoftDelete(ctx context.Context, id int64) error {
	xMonitorMetricsIdKey := fmt.Sprintf("%s%v", cacheXMonitorMetricsIdPrefix, id)
	_, err := m.ExecCtx(ctx, func(ctx context.Context, conn sqlx.SqlConn) (result sql.Result, err error) {
		query := fmt.Sprintf("update %s set deleted_at = ? where `id` = ?", m.table)
		return conn.ExecCtx(ctx, query, time.Now(), id)
	}, xMonitorMetricsIdKey)
	return err
}
func (m *defaultXMonitorMetricsModel) DeleteMetrics(ctx context.Context, periodType int64, dateFromTo string, employee string) error {
	total, reqList, err := m.List(ctx, &ListReq{
		PeriodType: periodType,
		DateFromTo: []string{dateFromTo},
		Employee:   employee,
	})
	if err != nil {
		return err
	}

	if total == 0 {
		return nil
	}

	return m.SoftDelete(ctx, reqList[0].Id)
}
