package okr

import (
	"context"
	"database/sql"
	"fmt"
	"strings"

	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type MsgContentAssign struct {
	EntityId        int64       `json:"entity_id"`
	ReqBy           string      `json:"req_by"`
	ReqHandleBy     string      `json:"req_handle_by"`
	ReqHandleResult int64       `json:"req_handle_result"` // req处理结果枚举值: 1拒绝 2接受并添加为o或kr
	EntityPath      []*XOKrTask `json:"entity_path"`
}

type MsgContentAssignReq struct {
	MsgContentAssign
}

type MsgContentAssignResp struct {
	MsgContentAssign
}

type MsgContentAlign struct {
	EntityId                  int64       `json:"entity_id"`
	AlignWithEntityId         int64       `json:"align_with_entity_id"`
	ReqBy                     string      `json:"req_by"`
	ReqHandleBy               string      `json:"req_handle_by"`
	ReqHandleResult           int64       `json:"req_handle_result"` // req处理结果枚举值: 1拒绝 2接受
	EntityPath                []*XOKrTask `json:"entity_path"`
	AlignWithEntityEntityPath []*XOKrTask `json:"align_with_entity_path"`
}

type MsgContentAlignReq struct {
	MsgContentAlign
}

type MsgContentAlignResp struct {
	MsgContentAlign
}

// 发起确认
type MsgConfirmRep struct {
	EntityId int64 `json:"entity_id"`
}

// 发起确认结果
type MsgConfirmResp struct {
	EntityId    int64  `json:"entity_id"`
	ReqBy       string `json:"req_by"`
	ReqHandleBy string `json:"req_handle_by"`
}

func (m *customXOKrTaskModel) GetByPeriod(ctx context.Context, session sqlx.Session, period int64) (list []*XOKrTask, err error) {
	args := []any{period}
	query := fmt.Sprintf(
		`
select * from %s where period_id = ?;
`,
		m.table)
	err = session.QueryRowsCtx(ctx, &list, query, args...)
	return list, err
}

func (m *customXOKrTaskModel) GetEntities(ctx context.Context, session sqlx.Session, entityIds []int64) (list []*XOKrTask, err error) {
	if len(entityIds) == 0 {
		return nil, nil
	}
	var placeholders []string
	var args []any
	for _, v := range entityIds {
		placeholders = append(placeholders, "?")
		args = append(args, v)
	}
	query := fmt.Sprintf(
		`
select *
from %s
where entity_id in (%s)
`,
		m.table,
		strings.Join(placeholders, ", "))

	err = session.QueryRowsCtx(ctx, &list, query, args...)
	return list, err
}

func (m *customXOKrTaskModel) GetEntityRootPaths(ctx context.Context, session sqlx.Session, entityIds []int64) (list []*XOKrTask, err error) {
	seen := make(map[int64]struct{})
	for {
		elems, err := m.GetEntities(ctx, session, entityIds)
		if err != nil {
			return nil, err
		}
		var newEntityIds []int64
		for _, elem := range elems {
			list = append(list, elem)
			seen[elem.EntityId] = struct{}{}
			if elem.EntityType == 1 {
				continue
			}
			if elem.ParentEntityId == 0 {
				continue
			}
			if _, ok := seen[elem.ParentEntityId]; ok {
				continue
			}
			newEntityIds = append(newEntityIds, elem.ParentEntityId)
		}
		if len(newEntityIds) == 0 {
			break
		}
		entityIds = newEntityIds
	}
	return list, err
}

func (m *customXOKrTaskModel) BulkUpdatePosition(ctx context.Context, session sqlx.Session, entityIdToPosition map[int64]int64) (sql.Result, error) {
	var entityIds []string
	var whenThenPairs []string
	for k, v := range entityIdToPosition {
		entityIds = append(entityIds, fmt.Sprint(k))
		whenThenPairs = append(whenThenPairs, fmt.Sprintf(" when %d then %d ", k, v))
	}
	query := fmt.Sprintf(
		`
update %s
set position = case entity_id %s else position end
where entity_id in (%s);
`,
		m.table,
		strings.Join(whenThenPairs, " "),
		strings.Join(entityIds, ", "))
	return session.ExecCtx(ctx, query)
}

func (m *customXOkrAlignmentModel) BulkUpdatePosition(ctx context.Context, session sqlx.Session, entityIdToPosition map[int64]int64) (sql.Result, error) {
	var entityIds []string
	var whenThenPairs []string
	for k, v := range entityIdToPosition {
		entityIds = append(entityIds, fmt.Sprint(k))
		whenThenPairs = append(whenThenPairs, fmt.Sprintf(" when %d then %d ", k, v))
	}
	query := fmt.Sprintf(
		`
update %s
set align_with_position = case entity_id %s else align_with_position end
where entity_id in (%s);
`,
		m.table,
		strings.Join(whenThenPairs, " "),
		strings.Join(entityIds, ", "))
	return session.ExecCtx(ctx, query)
}

func (m *customXOkrAlignmentModel) BulkSetAlignWithUser(ctx context.Context, session sqlx.Session, alignWithEntityId int64, alignWithUser string) (sql.Result, error) {
	query := fmt.Sprintf(
		`
update %s
set align_with_user = ?
where align_with_entity = ?;
`,
		m.table)
	return session.ExecCtx(ctx, query, alignWithUser, alignWithEntityId)
}

func (m *customXOkrAlignmentModel) GetAlignmentParents(ctx context.Context, session sqlx.Session, entityIds []int64, includePending bool) (list []*XOkrAlignment, err error) {
	if len(entityIds) == 0 {
		return nil, nil
	}
	var placeholders []string
	var args []any
	for _, v := range entityIds {
		placeholders = append(placeholders, "?")
		args = append(args, v)
	}
	query := fmt.Sprintf(
		`
select *
from %s
where entity_id in (%s)
`,
		m.table,
		strings.Join(placeholders, ", "))

	err = session.QueryRowsCtx(ctx, &list, query, args...)
	if err != nil {
		return nil, err
	}
	if !includePending {
		newList := make([]*XOkrAlignment, 0, len(list))
		for _, elem := range list {
			if elem.Intended == 1 {
				continue
			}
			newList = append(newList, elem)
		}
		list = newList
	}
	return list, err
}

func (m *customXOkrAlignmentModel) GetAlignmentChildren(ctx context.Context, session sqlx.Session, alignWithEntityIds []int64, includePending bool) (list []*XOkrAlignment, err error) {
	if len(alignWithEntityIds) == 0 {
		return nil, nil
	}
	var placeholders []string
	var args []any
	for _, v := range alignWithEntityIds {
		placeholders = append(placeholders, "?")
		args = append(args, v)
	}
	query := fmt.Sprintf(
		`
select *
from %s
where align_with_entity in (%s)
`,
		m.table,
		strings.Join(placeholders, ", "))

	err = session.QueryRowsCtx(ctx, &list, query, args...)
	if err != nil {
		return nil, err
	}
	if !includePending {
		newList := make([]*XOkrAlignment, 0, len(list))
		for _, elem := range list {
			if elem.Intended == 1 {
				continue
			}
			newList = append(newList, elem)
		}
		list = newList
	}
	return list, err
}

func (m *customXOkrAlignmentModel) GetAlignmentByAssign(ctx context.Context, session sqlx.Session, entityIds []int64) (list []*XOkrAlignment, err error) {
	if len(entityIds) == 0 {
		return nil, nil
	}
	var placeholders []string
	var args []any
	for _, v := range entityIds {
		placeholders = append(placeholders, "?")
		args = append(args, v)
	}
	query := fmt.Sprintf(
		`
select *
from %s
where entity_id in (%s) and by_assign = 1
`,
		m.table,
		strings.Join(placeholders, ", "))

	err = session.QueryRowsCtx(ctx, &list, query, args...)
	if err != nil {
		return nil, err
	}
	return list, err
}

func (m *customXOkrAlignmentModel) TransDelete(session sqlx.Session, ids []int64) (err error) {
	placeholders := make([]string, len(ids))
	var args []any
	for i := range placeholders {
		placeholders[i] = "?"
		args = append(args, ids[i])
	}
	query := fmt.Sprintf("delete from %s where `id` in (%s) ", m.table, strings.Join(placeholders, ","))
	_, err = session.Exec(query, args...)
	if err != nil {
		return err
	}
	return nil
}
