package okr

import (
	"context"
	"errors"
	"fmt"
	"strings"

	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type OkrAlignmentList []*XOkrAlignment

func (l OkrAlignmentList) GetByEntityId(entityId int64) *XOkrAlignment {
	for _, v := range l {
		if v.EntityId == entityId {
			return v
		}
	}
	return nil
}

func (l OkrAlignmentList) ByAlignWithEntityId() map[int64]OkrAlignmentList {
	m := make(map[int64]OkrAlignmentList)
	for _, v := range l {
		m[v.AlignWithEntity] = append(m[v.AlignWithEntity], v)
	}
	return m
}

func (m *defaultXOkrAlignmentModel) GetAlignedChildren(ctx context.Context, alignWithEntityIds []int64) (list OkrAlignmentList, err error) {
	if len(alignWithEntityIds) == 0 {
		return nil, errors.New("alignWithEntityIds is empty")
	}
	placeholders := make([]string, len(alignWithEntityIds))
	args := make([]any, len(alignWithEntityIds))
	for i, v := range alignWithEntityIds {
		placeholders[i] = "?"
		args[i] = v
	}

	query := fmt.Sprintf("select * from %s where `align_with_entity` in (%s) and intended = 0", m.table, strings.Join(placeholders, ","))
	err = m.conn.QueryRowsCtx(ctx, &list, query, args...)
	return
}

func (m *defaultXOkrAlignmentModel) GetByEntityId(ctx context.Context, entityId int64) (*XOkrAlignment, error) {
	query := fmt.Sprintf("select %s from %s where `entity_id` = ? limit 1", xOkrAlignmentRows, m.table)
	var resp XOkrAlignment
	err := m.conn.QueryRowCtx(ctx, &resp, query, entityId)
	switch err {
	case nil:
		return &resp, nil
	case sqlx.ErrNotFound:
		return nil, ErrNotFound
	default:
		return nil, err
	}
}

func (m *defaultXOkrAlignmentModel) TransUpdateCtx(ctx context.Context, session sqlx.Session, newData *XOkrAlignment) error {
	query := fmt.Sprintf("update %s set %s where `id` = ?", m.table, xOkrAlignmentRowsWithPlaceHolder)
	_, err := session.ExecCtx(ctx, query, newData.EntityId, newData.AlignWithEntity, newData.AlignWithUser, newData.AlignWithPosition, newData.AlignWithWeight, newData.Intended, 0, newData.Id)
	return err
}

func (m *defaultXOkrAlignmentModel) TransGetOneByUserAndAlignWithEntityId(ctx context.Context, session sqlx.Session, userEmail string, alignWithEntityId, entityId int64) (*XOkrAlignment, error) {
	query := fmt.Sprintf("select * from %s where `align_with_user` = ? and `align_with_entity` = ? and `entity_id` = ? and `intended` = ? limit 1", m.table)
	var row XOkrAlignment
	err := session.QueryRowCtx(ctx, &row, query, userEmail, alignWithEntityId, entityId, 0)
	switch err {
	case nil:
		return &row, nil
	case sqlx.ErrNotFound:
		return nil, ErrNotFound
	default:
		return nil, err
	}
}

// 查询用户某个entity_id下的所有对齐
func (m *defaultXOkrAlignmentModel) FindAllByUserAndAlignWithEntitiyId(ctx context.Context, userEmail string, alignWithEntityId int64) (list OkrAlignmentList, err error) {
	query := fmt.Sprintf("select * from %s where `align_with_user` = ? and `align_with_entity` = ? and `intended` = ?", m.table)
	err = m.conn.QueryRowsCtx(ctx, &list, query, userEmail, alignWithEntityId, 0)
	return
}

func (m *defaultXOkrAlignmentModel) FindByAlignWithEntityId(alignWithEntityId int64) ([]*XOkrAlignment, error) {
	var resp []*XOkrAlignment
	query := fmt.Sprintf("select * from %s where `align_with_entity` = ? ", m.table)
	err := m.conn.QueryRows(&resp, query, alignWithEntityId)
	if err != nil {
		return nil, err
	}
	return resp, nil
}

func (m *defaultXOkrAlignmentModel) DeleteByIds(ids []int64) error {
	placeholders := make([]string, len(ids))
	var args []any
	for i := range placeholders {
		placeholders[i] = "?"
		args = append(args, ids[i])
	}
	query := fmt.Sprintf("delete from %s where `id` in (%s) ", m.table, strings.Join(placeholders, ","))
	_, err := m.conn.Exec(query, args...)
	if err != nil {
		return err
	}
	return nil
}

// 事务删除 删除某个entity_id下的所有对齐
func (m *defaultXOkrAlignmentModel) DeleteByAlignWithEntityIdAndUserEmail(ctx context.Context, session sqlx.Session, entityId int64, userEmail string) error {
	query := fmt.Sprintf("delete from %s where `align_with_entity` = ? and `align_with_user` = ?", m.table)
	_, err := session.ExecCtx(ctx, query, entityId, userEmail)
	return err
}
func (m *defaultXOkrAlignmentModel) DeleteByEntityId(ctx context.Context, session sqlx.Session, entityId int64) error {
	query := fmt.Sprintf("delete from %s where `entity_id` = ?", m.table)
	_, err := session.ExecCtx(ctx, query, entityId)
	return err
}

func (m *defaultXOkrAlignmentModel) GetAlignWithEntityIdAndAlignID(ctx context.Context, entityId int64, alignID int64) (list OkrAlignmentList, err error) {
	query := fmt.Sprintf("select * from %s where `entity_id` = ? and `align_with_entity` = ?", m.table)
	err = m.conn.QueryRowsCtx(ctx, &list, query, entityId, alignID)
	return
}
