package okr

import (
	"context"
	"database/sql"
	"fmt"
	"strings"

	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type OKrTaskList []*XOKrTask

func (l OKrTaskList) GetByEntityId(entityId int64) *XOKrTask {
	for _, v := range l {
		if v.EntityId == entityId {
			return v
		}
	}
	return nil
}

func (m *defaultXOKrTaskModel) GetOneUserOkrEntity(ctx context.Context, periodId, parentEntityId, entityId int64) (*XOKrTask, error) {
	query := fmt.Sprintf("select * from %s where `entity_id` = ? and `parent_entity_id` = ? and `period_id` = ? limit 1", m.table)
	var entity XOKrTask
	err := m.conn.QueryRowCtx(ctx, &entity, query, entityId, parentEntityId, periodId)
	switch err {
	case nil:
		return &entity, nil
	case sqlx.ErrNotFound:
		return nil, ErrNotFound
	default:
		return nil, err
	}
}

// 获取用户某个entity下面所有的对齐entity
func (m *defaultXOKrTaskModel) FindByUserAndAlignWithEntity(ctx context.Context, owner string, alignWithEntityId int64) (list OKrTaskList, err error) {
	query := fmt.Sprintf(`select a.* from %s a left join x_okr_alignment b 
	on a.entity_id = b.entity_id 
	where b.align_with_user = ? and b.align_with_entity = ?`, m.table)
	err = m.conn.QueryRowsCtx(ctx, &list, query, owner, alignWithEntityId)
	return
}

func (m *defaultXOKrTaskModel) GetChildren(parentID int64) (list []*XOKrTask, err error) {
	err = m.conn.QueryRows(&list, fmt.Sprintf("select * from %s where parent_entity_id = ?", m.table), parentID)
	if err != nil {
		return nil, err
	}
	return list, nil
}

// 自定义事务方法
func (m *customXOKrTaskModel) TransInsertCtx(ctx context.Context, session sqlx.Session, data *XOKrTask) (sql.Result, error) {
	query := fmt.Sprintf("insert into %s (%s) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)", m.table, xOKrTaskRowsExpectAutoSet)
	ret, err := session.ExecCtx(ctx, query, data.EntityType, data.ParentEntityId, data.PeriodId, data.Content, data.Priority, data.Deadline, data.TypeForKr, data.Owner, data.Position, data.Weight, data.IntendedOwner)
	return ret, err
}

func (m *customXOKrTaskModel) TransUpdateCtx(ctx context.Context, session sqlx.Session, data *XOKrTask) error {
	query := fmt.Sprintf("update %s set %s where `entity_id` = ?", m.table, xOKrTaskRowsWithPlaceHolder)
	_, err := session.ExecCtx(ctx, query, data.EntityType, data.ParentEntityId, data.PeriodId, data.Content, data.Priority, data.Deadline, data.TypeForKr, data.Owner, data.Position, data.Weight, data.IntendedOwner, data.EntityId)
	return err
}

func (m *customXOKrTaskModel) TransDeleteCtx(ctx context.Context, session sqlx.Session, entityIds []int64) error {
	if len(entityIds) == 0 {
		return nil // 若切片为空，直接返回 nil，避免执行无意义查询
	}
	// 生成对应数量的占位符，例如 "?, ?, ?, ?"
	placeholders := strings.Repeat(",?", len(entityIds))[1:]
	query := fmt.Sprintf("delete from %s where `entity_id` in (%s)", m.table, placeholders)
	// 将切片元素展开作为参数
	args := make([]interface{}, len(entityIds))
	for i, id := range entityIds {
		args[i] = id
	}
	_, err := session.ExecCtx(ctx, query, args...)
	return err
}

func (m *customXOKrTaskModel) TransCtx(ctx context.Context, fn func(ctx context.Context, s sqlx.Session) error) error {
	return m.conn.TransactCtx(ctx, func(ctx context.Context, s sqlx.Session) error {
		return fn(ctx, s)
	})
}

func (m *defaultXOKrTaskModel) GetByOkrPeriodId(okrPeriodId int64) (list []*XOKrTask, err error) {
	err = m.conn.QueryRows(&list, fmt.Sprintf("select * from %s where period_id = ?", m.table), okrPeriodId)
	if err != nil {
		return nil, err
	}
	return list, nil
}
func (m *defaultXOKrTaskModel) GetByOkrPeriodIdAndEntityType(okrPeriodId, entityType int64) (list OKrTaskList, err error) {
	err = m.conn.QueryRows(&list, fmt.Sprintf("select * from %s where period_id = ? and entity_type = ?", m.table), okrPeriodId, entityType)
	if err != nil {
		return nil, err
	}
	return list, nil
}
func (m *defaultXOKrTaskModel) GetByParentEntityId(entityId int64) (list OKrTaskList, err error) {
	err = m.conn.QueryRows(&list, fmt.Sprintf("select * from %s where parent_entity_id = ?", m.table), entityId)
	if err != nil {
		return nil, err
	}
	return list, nil
}

func (m *defaultXOKrTaskModel) InsertWithSession(conn sqlx.Session, data *XOKrTask) (sql.Result, error) {
	query := fmt.Sprintf("insert into %s (entity_type,parent_entity_id,period_id,content,priority,deadline,type_for_kr,owner,position,weight,intended_owner) values (?,?, ?, ?, ?, ?, ?, ?, ?, ?, ?)", m.table)
	return conn.Exec(query, data.EntityType, data.ParentEntityId, data.PeriodId, data.Content, data.Priority,
		data.Deadline, data.TypeForKr, data.Owner, data.Position, data.Weight, data.IntendedOwner)
}
func (m *defaultXOKrTaskModel) UpdateWithSession(conn sqlx.Session, data *XOKrTask) (sql.Result, error) {
	query := fmt.Sprintf("update %s set content = ?,priority = ?,deadline = ?,weight = ?,intended_owner = ? where entity_id = ?", m.table)
	return conn.Exec(query, data.Content, data.Priority, data.Deadline, data.Weight, data.IntendedOwner, data.EntityId)
}
