package performance

import (
	"context"
	"database/sql"
	"errors"
	"fmt"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/globalkey"
	"strings"
	"time"

	"github.com/zeromicro/go-zero/core/stores/sqlc"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

func (m *customXPerformanceModel) XDelete(ctx context.Context, id uint64) error {
	query := fmt.Sprintf("update %s set deleted_at = ? where `id` = ?", m.table)
	_, err := m.conn.ExecCtx(ctx, query, time.Now(), id)
	return err
}

func (m *customXPerformanceModel) XFindOne(ctx context.Context, id uint64) (*XPerformance, error) {
	query := fmt.Sprintf("select %s from %s where `id` = ? %s limit 1", xPerformanceRows, m.table, queryBuilder())
	var resp XPerformance
	err := m.conn.QueryRowCtx(ctx, &resp, query, id)
	switch err {
	case nil:
		return &resp, nil
	case sqlx.ErrNotFound:
		return nil, ErrNotFound
	default:
		return nil, err
	}
}
func (m *customXPerformanceModel) XFindOneByIds(ctx context.Context, ids []uint64) ([]*XPerformance, error) {
	//query := fmt.Sprintf("select %s from %s where `id` in ? (%s) ", xPerformanceRows, m.table, queryBuilder())
	var resp []*XPerformance
	//err := m.conn.QueryRowsCtx(ctx, &resp, query, strings.Repeat(",?", len(ids)[1:]))
	conditions := []string{fmt.Sprintf("`id` in (%s)", strings.Repeat(",?", len(ids))[1:])}
	var args []any
	for _, v := range ids {
		args = append(args, v)
	}
	query := fmt.Sprintf("select * from %s where %s", m.table, strings.Join(conditions, " and "))
	err := m.conn.QueryRowsCtx(ctx, &resp, query, args...)
	switch err {
	case nil:
		return resp, nil
	case sqlx.ErrNotFound:
		return nil, ErrNotFound
	default:
		return nil, err
	}
}

func (m *defaultXPerformanceModel) XInsert(ctx context.Context, data *XPerformance) (sql.Result, error) {
	query := fmt.Sprintf("insert into %s (%s) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)", m.table, xPerformanceRowsExpectAutoSet)
	ret, err := m.conn.ExecCtx(ctx, query, data.UserEmail, data.LeaderEmail, data.Year, data.MonthNum, data.WorkItemWeight, data.AdditionItemWeight, data.TotalScore, data.LeaderWords, data.Status, data.TimeFactor, data.Level, data.LevelFactor, data.IncidentDeduction, data.ExcellentMemberBonus, data.ExcellentManagerBonus, data.ProjectBonus, data.TotalBonus, data.IsExcellentEmployee, data.ExcellentEmployeeReason, data.DeletedAt)
	return ret, err
}

func (m *defaultXPerformanceModel) XUpdate(ctx context.Context, data *XPerformance) error {
	query := fmt.Sprintf("update %s set %s where `id` = ? %s", m.table, xPerformanceRowsWithPlaceHolder, queryBuilder())
	_, err := m.conn.ExecCtx(ctx, query, data.UserEmail, data.LeaderEmail, data.Year, data.MonthNum, data.WorkItemWeight, data.AdditionItemWeight, data.TotalScore, data.LeaderWords, data.Status, data.TimeFactor, data.Level, data.LevelFactor, data.IncidentDeduction, data.ExcellentMemberBonus, data.ExcellentManagerBonus, data.ProjectBonus, data.TotalBonus, data.IsExcellentEmployee, data.ExcellentEmployeeReason, data.DeletedAt, data.Id)
	return err
}

type UpdatePerformanceBonusQuery struct {
	PerformanceID           uint64
	LevelFactor             float64
	TimeFactor              float64
	IncidentDeduction       float64
	ExcellentMemberBonus    float64
	ExcellentManagerBonus   float64
	TotalBonus              float64
	IsExcellentEmployee     uint64
	ExcellentEmployeeReason string
}

func (m *customXPerformanceModel) UpdatePerformanceBonus(ctx context.Context, req UpdatePerformanceBonusQuery) error {
	query := fmt.Sprintf("update %s set `level_factor` = ?, `time_factor` = ?,`incident_deduction` = ?, `excellent_member_bonus` = ?, `excellent_manager_bonus` = ?, `total_bonus` = ?,`is_excellent_employee` = ?,`excellent_employee_reason` = ? where `id` = ? %s", m.table, queryBuilder())
	_, err := m.conn.ExecCtx(ctx, query, req.LevelFactor, req.TimeFactor, req.IncidentDeduction, req.ExcellentMemberBonus, req.ExcellentManagerBonus, req.TotalBonus, req.IsExcellentEmployee, req.ExcellentEmployeeReason, req.PerformanceID)
	return err
}

type UpdatePerformanceStatusQuery struct {
	PerformanceID uint64
	Status        uint64
}

func (m *customXPerformanceModel) UpdatePerformanceStatus(ctx context.Context, req UpdatePerformanceStatusQuery) error {
	query := fmt.Sprintf("update %s set  `status` = ? where `id` = ?", m.table)
	_, err := m.conn.ExecCtx(ctx, query, req.Status, req.PerformanceID)
	return err
}

// 自定义事务方法
func (m *customXPerformanceModel) TransInsertCtx(ctx context.Context, session sqlx.Session, data *XPerformance) (sql.Result, error) {
	query := fmt.Sprintf("insert into %s (%s) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)", m.table, xPerformanceRowsExpectAutoSet)
	ret, err := session.ExecCtx(ctx, query, data.UserEmail, data.LeaderEmail, data.Year, data.MonthNum, data.WorkItemWeight, data.AdditionItemWeight, data.TotalScore, data.LeaderWords, data.Status, data.TimeFactor, data.Level, data.LevelFactor, data.IncidentDeduction, data.ExcellentMemberBonus, data.ExcellentManagerBonus, data.ProjectBonus, data.TotalBonus, data.IsExcellentEmployee, data.ExcellentEmployeeReason, data.DeletedAt)
	return ret, err
}

func (m *customXPerformanceModel) TransUpdateCtx(ctx context.Context, session sqlx.Session, data *XPerformance) error {
	query := fmt.Sprintf("update %s set %s where `id` = ? %s", m.table, xPerformanceRowsWithPlaceHolder, queryBuilder())
	_, err := session.ExecCtx(ctx, query, data.UserEmail, data.LeaderEmail, data.Year, data.MonthNum, data.WorkItemWeight, data.AdditionItemWeight, data.TotalScore, data.LeaderWords, data.Status, data.TimeFactor, data.Level, data.LevelFactor, data.IncidentDeduction, data.ExcellentMemberBonus, data.ExcellentManagerBonus, data.ProjectBonus, data.TotalBonus, data.IsExcellentEmployee, data.ExcellentEmployeeReason, data.DeletedAt, data.Id)
	return err
}

func (m *customXPerformanceModel) TransDeleteCtx(ctx context.Context, session sqlx.Session, id uint64) error {
	query := fmt.Sprintf("update %s  set deleted_at = ? where `id` = ?", m.table)
	_, err := session.ExecCtx(ctx, query, time.Now(), id)
	return err
}

func (m *customXPerformanceModel) TransCtx(ctx context.Context, fn func(ctx context.Context, s sqlx.Session) error) error {
	return m.conn.TransactCtx(ctx, func(ctx context.Context, s sqlx.Session) error {
		return fn(ctx, s)
	})
}

func (m *customXPerformanceModel) GetOneByUserAndTime(ctx context.Context, userEmail string, year, monthNum uint64) (*XPerformance, error) {
	var p XPerformance
	query := fmt.Sprintf("select * from %s where `year` = ? and month_num = ? and user_email = ? %s limit 1", m.table, queryBuilder())
	err := m.conn.QueryRowCtx(ctx, &p, query, year, monthNum, userEmail)
	switch err {
	case nil:
		return &p, nil
	case sqlc.ErrNotFound:
		return nil, ErrNotFound
	default:
		return nil, err
	}
}
func (m *customXPerformanceModel) GetOne(ctx context.Context, userEmail string, year, monthNum, isExcellentEmployee uint64) (*XPerformance, error) {
	var p XPerformance
	var err error
	if isExcellentEmployee > 0 {
		query := fmt.Sprintf("select * from %s where `year` =? and month_num =? and user_email =? and is_excellent_employee =? %s limit 1", m.table, queryBuilder())
		err = m.conn.QueryRowCtx(ctx, &p, query, year, monthNum, userEmail, isExcellentEmployee)
	} else {
		query := fmt.Sprintf("select * from %s where `year` = ? and month_num = ? and user_email = ? %s limit 1", m.table, queryBuilder())
		err = m.conn.QueryRowCtx(ctx, &p, query, year, monthNum, userEmail)
	}
	switch err {
	case nil:
		return &p, nil
	case sqlc.ErrNotFound:
		return nil, ErrNotFound
	default:
		return nil, err
	}
}

type FindAllByUserAndTimeQuery struct {
	Year                uint64
	MonthNum            uint64
	SortField           string
	SortOrder           string
	PageNum             uint64
	PageSize            uint64
	EmailList           []string
	IsExcellentEmployee uint64
	Status              uint64
}

func (m *customXPerformanceModel) FindAllByUserAndTime(ctx context.Context, req FindAllByUserAndTimeQuery) (PerformanceList, error) {
	if len(req.EmailList) == 0 {
		return nil, errors.New("email list is empty")
	}

	conditions := []string{fmt.Sprintf("`user_email` in (%s)", strings.Repeat(",?", len(req.EmailList))[1:])}
	var args []any
	for _, v := range req.EmailList {
		args = append(args, v)
	}

	conditions = append(conditions, "`year` = ?")
	args = append(args, req.Year)
	if req.MonthNum > 0 {
		conditions = append(conditions, "`month_num` = ?")
		args = append(args, req.MonthNum)
	}
	if req.IsExcellentEmployee > 0 {
		conditions = append(conditions, "`is_excellent_employee` = ?")
		args = append(args, req.IsExcellentEmployee)
	}
	if req.Status > 0 {
		if req.Status == uint64(oa.PerformanceStatus_IN_EVALUATION) { //49 代表环评中的，也就是没有结束的  不是5,50,51的
			conditions = append(conditions, "`status` not in (5,50,51)")
		} else {
			conditions = append(conditions, "`status` = ?")
			args = append(args, req.Status)
		}
	}
	conditions = append(conditions, deletedAtIsNull)

	query := fmt.Sprintf("select * from %s where %s", m.table, strings.Join(conditions, " and "))
	if req.SortField != "" && req.SortOrder != "" {
		query = fmt.Sprintf("%s order by %s %s", query, req.SortField, req.SortOrder)
	}

	if req.PageNum > 0 && req.PageSize > 0 {
		offset := req.PageSize * (req.PageNum - 1)
		query = fmt.Sprintf("%s limit ? offset ?", query)
		args = append(args, req.PageSize, offset)
	}

	var list PerformanceList
	err := m.conn.QueryRowsCtx(ctx, &list, query, args...)
	if err != nil {
		return nil, err
	}
	return list, nil
}

type RankingCountData struct {
	Total                 uint64               `db:"total"`
	TotalBonus            float64              `db:"total_bonus"`
	RankingCountLevelList []*RankingCountLevel `db:"ranking_count_level_list"`
}
type RankingCountLevel struct {
	Level   string  `db:"level"`
	Count   uint64  `db:"count"`
	Percent float64 `db:"percent"`
}

func (m *customXPerformanceModel) CountByUserAndTime(ctx context.Context, emails []string, year, monthNum uint64, isExcellentEmployee uint64, status uint64) (*RankingCountData, error) {
	if len(emails) == 0 {
		return nil, errors.New("emails is empty")
	}
	conditions := []string{
		fmt.Sprintf("`user_email` in (%s)", strings.Repeat(",?", len(emails))[1:]),
		"`year` = ?",
		"`month_num` =?",
		deletedAtIsNull,
	}
	var args []any
	for _, v := range emails {
		args = append(args, v)
	}
	args = append(args, year, monthNum)

	if isExcellentEmployee > 0 {
		conditions = append(conditions, "`is_excellent_employee` = ?")
		args = append(args, isExcellentEmployee)
	}
	if status > 0 {
		if status == uint64(oa.PerformanceStatus_IN_EVALUATION) { //49 代表环评中的，也就是没有结束的  不是5,50,51的
			conditions = append(conditions, "`status` not in (5,50,51)")
		} else {
			conditions = append(conditions, "`status` = ?")
			args = append(args, status)
		}
	}
	query := fmt.Sprintf("select count(*) as `total`, IFNULL(sum(`total_bonus`),0) as `total_bonus` from %s where %s", m.table, strings.Join(conditions, " and "))
	var data RankingCountData
	if err := m.conn.QueryRowPartialCtx(ctx, &data, query, args...); err != nil {
		return nil, err
	}
	queryLevel := fmt.Sprintf("select level,count(*) as `count` from %s where %s group by level order by level", m.table, strings.Join(conditions, " and "))

	if err := m.conn.QueryRowsPartialCtx(ctx, &data.RankingCountLevelList, queryLevel, args...); err != nil {
		return nil, err
	}
	return &data, nil
}

func (m *customXPerformanceModel) CountByTime(ctx context.Context, year, monthNum uint64) (*RankingCountData, error) {
	query := fmt.Sprintf("select count(*) as `total`, IFNULL(sum(`total_bonus`),0) as `total_bonus` from %s where `year` = ? and `month_num` = ? %s", m.table, queryBuilder())
	var data RankingCountData
	if err := m.conn.QueryRowPartialCtx(ctx, &data, query, year, monthNum); err != nil {
		return nil, err
	}
	return &data, nil
}

type FindAllByTimeQuery struct {
	Year      uint64
	MonthNum  uint64
	SortField string
	SortOrder string
	PageNum   uint64
	PageSize  uint64
}

func (m *customXPerformanceModel) FindAllByTime(ctx context.Context, req FindAllByTimeQuery) (PerformanceList, error) {
	condition := []string{"`year` = ?"}
	args := []any{req.Year}
	if req.MonthNum > 0 {
		condition = append(condition, "`month_num` = ?")
		args = append(args, req.MonthNum)
	}
	condition = append(condition, deletedAtIsNull)

	query := fmt.Sprintf("select * from %s where %s", m.table, strings.Join(condition, " and "))
	if req.SortField != "" && req.SortOrder != "" {
		query = fmt.Sprintf("%s order by %s %s", query, req.SortField, req.SortOrder)
	}
	if req.PageNum > 0 && req.PageSize > 0 {
		offset := req.PageSize * (req.PageNum - 1)
		query = fmt.Sprintf("%s limit ? offset ?", query)
		args = append(args, req.PageSize, offset)
	}
	var list PerformanceList
	err := m.conn.QueryRowsCtx(ctx, &list, query, args...)
	if err != nil {
		return nil, err
	}
	return list, nil
}

func (m *customXPerformanceModel) FindByUserAndTime(ctx context.Context, userEmail string, year, month_num uint64) (PerformanceList, error) {
	var list PerformanceList
	var err error
	if month_num > 0 {
		query := fmt.Sprintf("select * from %s where `year` = ? and user_email = ? and month_num = ? %s", m.table, queryBuilder())
		err = m.conn.QueryRowsCtx(ctx, &list, query, year, userEmail, month_num)
	} else {
		query := fmt.Sprintf("select * from %s where `year` = ? and user_email = ? %s", m.table, queryBuilder())
		err = m.conn.QueryRowsCtx(ctx, &list, query, year, userEmail)
	}
	return list, err
}

type PerformanceList []*XPerformance

func (l PerformanceList) Ids() []uint64 {
	ids := make([]uint64, len(l))
	for i, v := range l {
		ids[i] = v.Id
	}
	return ids
}

// userEmail对当前的绩效是否可查看
// 绩效只有自己，直接领导，Rocky可查看
func (p *XPerformance) CanView(userEmail, leaderEmail string) bool {
	switch userEmail {
	case p.UserEmail, leaderEmail, globalkey.RockyEmail:
		return true
	}
	return false
}
