package performance

import (
	"context"
	"database/sql"
	"fmt"
	"time"

	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

func (m *customXPerformanceItemCommentModel) XDelete(ctx context.Context, id uint64) error {
	query := fmt.Sprintf("update %s set `deleted_at` = ? where `id` = ?", m.table)
	_, err := m.conn.ExecCtx(ctx, query, time.Now(), id)
	return err
}

func (m *customXPerformanceItemCommentModel) XFindOne(ctx context.Context, id uint64) (*XPerformanceItemComment, error) {
	query := fmt.Sprintf("select %s from %s where `id` = ? %s limit 1", xPerformanceItemCommentRows, m.table, queryBuilder())
	var resp XPerformanceItemComment
	err := m.conn.QueryRowCtx(ctx, &resp, query, id)
	switch err {
	case nil:
		return &resp, nil
	case sqlx.ErrNotFound:
		return nil, ErrNotFound
	default:
		return nil, err
	}
}

func (m *customXPerformanceItemCommentModel) XInsert(ctx context.Context, data *XPerformanceItemComment) (sql.Result, error) {
	query := fmt.Sprintf("insert into %s (%s) values (?, ?, ?, ?, ?, ?)", m.table, xPerformanceItemCommentRowsExpectAutoSet)
	ret, err := m.conn.ExecCtx(ctx, query, data.PerformanceId, data.ItemId, data.CommentType, data.UserEmail, data.Comment, data.DeletedAt)
	return ret, err
}

func (m *customXPerformanceItemCommentModel) XUpdate(ctx context.Context, data *XPerformanceItemComment) error {
	query := fmt.Sprintf("update %s set %s where `id` = ? %s", m.table, xPerformanceItemCommentRowsWithPlaceHolder, queryBuilder())
	_, err := m.conn.ExecCtx(ctx, query, data.PerformanceId, data.ItemId, data.CommentType, data.UserEmail, data.Comment, data.DeletedAt, data.Id)
	return err
}

func (m *customXPerformanceItemCommentModel) TransUpdateCtx(ctx context.Context, session sqlx.Session, data *XPerformanceItemComment) error {
	query := fmt.Sprintf("update %s set %s where `id` = ? %s", m.table, xPerformanceItemCommentRowsWithPlaceHolder, queryBuilder())
	_, err := session.ExecCtx(ctx, query, data.PerformanceId, data.ItemId, data.CommentType, data.UserEmail, data.Comment, data.DeletedAt, data.Id)
	return err
}

func (m *customXPerformanceItemCommentModel) TransInsertCtx(ctx context.Context, session sqlx.Session, data *XPerformanceItemComment) (sql.Result, error) {
	query := fmt.Sprintf("insert into %s (%s) values (?, ?, ?, ?, ?, ?)", m.table, xPerformanceItemCommentRowsExpectAutoSet)
	ret, err := session.ExecCtx(ctx, query, data.PerformanceId, data.ItemId, data.CommentType, data.UserEmail, data.Comment, data.DeletedAt)
	return ret, err
}

func (m *customXPerformanceItemCommentModel) TransDeleteCtx(ctx context.Context, session sqlx.Session, id uint64) error {
	query := fmt.Sprintf("update %s set `deleted_at` = ? where `id` = ?", m.table)
	_, err := session.ExecCtx(ctx, query, time.Now(), id)
	return err
}

func (m *customXPerformanceItemCommentModel) TransMultiDeleteCtx(ctx context.Context, session sqlx.Session, ids []uint64) error {
	query := fmt.Sprintf("update %s set `deleted_at` = ? where `id` in (%s) %s ", m.table, formatIds(ids), queryBuilder())
	_, err := session.ExecCtx(ctx, query, time.Now())
	return err
}

func (m *customXPerformanceItemCommentModel) TransCtx(ctx context.Context, fn func(ctx context.Context, s sqlx.Session) error) error {
	return m.conn.TransactCtx(ctx, func(ctx context.Context, s sqlx.Session) error {
		return fn(ctx, s)
	})
}

func (m *customXPerformanceItemCommentModel) FindByPerformanceItem(ctx context.Context, performanceId, itemId uint64, commentType uint32) (PerformanceItemCommentList, error) {
	query := fmt.Sprintf("select * from %s where `performance_id` = ? and `item_id` = ? and `comment_type` = ? %s order by `id` asc", m.table, queryBuilder())
	var rows PerformanceItemCommentList
	err := m.conn.QueryRowsCtx(ctx, &rows, query, performanceId, itemId, commentType)
	return rows, err
}

func (m *customXPerformanceItemCommentModel) FindByPerformance(ctx context.Context, performanceId uint64) (PerformanceItemCommentList, error) {
	query := fmt.Sprintf("select * from %s where `performance_id` = ? %s order by `id` asc", m.table, queryBuilder())
	var rows PerformanceItemCommentList
	err := m.conn.QueryRowsCtx(ctx, &rows, query, performanceId)
	return rows, err
}

type PageByPerformanceItemReq struct {
	PerformanceId uint64
	ItemId        uint64
	CommentType   uint32
	PageNum       uint64
	PageSize      uint64
	OrderBy       string // "updated_at desc"
}

func (m *customXPerformanceItemCommentModel) PageByPerformanceItem(ctx context.Context, req PageByPerformanceItemReq) (PerformanceItemCommentList, error) {
	offset := req.PageSize * (req.PageNum - 1)
	query := fmt.Sprintf("select * from %s where `performance_id` = ? and `item_id` = ? and `comment_type` = ? %s limit ? offset ?", m.table, queryBuilder())
	if req.OrderBy != "" {
		query = fmt.Sprintf("select * from %s where `performance_id` = ? and `item_id` = ? and `comment_type` = ? %s order by %s limit ? offset ?", m.table, queryBuilder(), req.OrderBy)
	}
	var rows PerformanceItemCommentList
	err := m.conn.QueryRowsCtx(ctx, &rows, query, req.PerformanceId, req.ItemId, req.CommentType, req.PageSize, offset)
	return rows, err
}

type PerformanceItemCommentList []*XPerformanceItemComment

func (l PerformanceItemCommentList) Ids() []uint64 {
	ids := make([]uint64, len(l))
	for i, v := range l {
		ids[i] = v.Id
	}
	return ids
}

func (l PerformanceItemCommentList) ByItemIdAndCommentType(itemId, commentType uint64) PerformanceItemCommentList {
	var list PerformanceItemCommentList
	for _, v := range l {
		if v.ItemId == itemId && v.CommentType == commentType {
			list = append(list, v)
		}
	}
	return list
}
