package performance

import (
	"context"
	"database/sql"
	"fmt"
	"time"

	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

func (m *customXPerformanceWorkModel) XDelete(ctx context.Context, id uint64) error {
	query := fmt.Sprintf("update %s set `deleted_at` = ? where `id` = ?", m.table)
	_, err := m.conn.ExecCtx(ctx, query, time.Now(), id)
	return err
}

func (m *customXPerformanceWorkModel) XFindOne(ctx context.Context, id uint64) (*XPerformanceWork, error) {
	query := fmt.Sprintf("select %s from %s where `id` = ? %s limit 1", xPerformanceWorkRows, m.table, queryBuilder())
	var resp XPerformanceWork
	err := m.conn.QueryRowCtx(ctx, &resp, query, id)
	switch err {
	case nil:
		return &resp, nil
	case sqlx.ErrNotFound:
		return nil, ErrNotFound
	default:
		return nil, err
	}
}

func (m *customXPerformanceWorkModel) XInsert(ctx context.Context, data *XPerformanceWork) (sql.Result, error) {
	query := fmt.Sprintf("insert into %s (%s) values (?, ?, ?, ?, ?, ?, ?, ?)", m.table, xPerformanceWorkRowsExpectAutoSet)
	ret, err := m.conn.ExecCtx(ctx, query, data.PerformanceId, data.ItemName, data.ItemWeight, data.ItemDesc, data.ItemDeficiency, data.SelfScore, data.LeaderScore, data.DeletedAt)
	return ret, err
}

func (m *customXPerformanceWorkModel) XUpdate(ctx context.Context, data *XPerformanceWork) error {
	query := fmt.Sprintf("update %s set %s where `id` = ? %s ", m.table, xPerformanceWorkRowsWithPlaceHolder, queryBuilder())
	_, err := m.conn.ExecCtx(ctx, query, data.PerformanceId, data.ItemName, data.ItemWeight, data.ItemDesc, data.ItemDeficiency, data.SelfScore, data.LeaderScore, data.DeletedAt, data.Id)
	return err
}

// 自定义事务方法
func (m *customXPerformanceWorkModel) TransInsertCtx(ctx context.Context, session sqlx.Session, data *XPerformanceWork) (sql.Result, error) {
	query := fmt.Sprintf("insert into %s (%s) values (?, ?, ?, ?, ?, ?, ?, ?)", m.table, xPerformanceWorkRowsExpectAutoSet)
	ret, err := session.ExecCtx(ctx, query, data.PerformanceId, data.ItemName, data.ItemWeight, data.ItemDesc, data.ItemDeficiency, data.SelfScore, data.LeaderScore, data.DeletedAt)
	return ret, err
}

func (m *customXPerformanceWorkModel) TransUpdateCtx(ctx context.Context, session sqlx.Session, data *XPerformanceWork) error {
	query := fmt.Sprintf("update %s set %s where `id` = ? %s", m.table, xPerformanceWorkRowsWithPlaceHolder, queryBuilder())
	_, err := session.ExecCtx(ctx, query, data.PerformanceId, data.ItemName, data.ItemWeight, data.ItemDesc, data.ItemDeficiency, data.SelfScore, data.LeaderScore, data.DeletedAt, data.Id)
	return err
}

func (m *customXPerformanceWorkModel) TransDeleteCtx(ctx context.Context, session sqlx.Session, id uint64) error {
	query := fmt.Sprintf("update %s  set `deleted_at` = ? where `id` = ?", m.table)
	_, err := session.ExecCtx(ctx, query, time.Now(), id)
	return err
}

func (m *customXPerformanceWorkModel) TransCtx(ctx context.Context, fn func(ctx context.Context, s sqlx.Session) error) error {
	return m.conn.TransactCtx(ctx, func(ctx context.Context, s sqlx.Session) error {
		return fn(ctx, s)
	})
}

func (m *customXPerformanceWorkModel) GetWorkItemsByPerformanceId(ctx context.Context, performanceId uint64) (PerformanceWorkItemList, error) {
	var list PerformanceWorkItemList
	query := fmt.Sprintf("select * from %s where `performance_id` = ? %s order by `id` asc", m.table, queryBuilder())
	err := m.conn.QueryRowsCtx(ctx, &list, query, performanceId)
	if err != nil {
		return nil, err
	}
	return list, nil
}

func (m *customXPerformanceWorkModel) TransGetWorkItemsByPerformanceId(ctx context.Context, session sqlx.Session, performanceId uint64) (PerformanceWorkItemList, error) {
	var list PerformanceWorkItemList
	query := fmt.Sprintf("select * from %s where `performance_id` = ? %s order by `id` asc", m.table, queryBuilder())
	err := session.QueryRowsCtx(ctx, &list, query, performanceId)
	if err != nil {
		return nil, err
	}
	return list, nil
}

func (m *customXPerformanceWorkModel) GetWorkItemsByPerformanceIds(ctx context.Context, performanceIds []uint64) (PerformanceWorkItemList, error) {
	var list PerformanceWorkItemList
	query := fmt.Sprintf("select * from %s where `performance_id` in (%s) %s order by `id` asc", m.table, formatIds(performanceIds), queryBuilder())
	err := m.conn.QueryRowsCtx(ctx, &list, query)
	if err != nil {
		return nil, err
	}
	return list, nil
}

type PerformanceWorkItemList []*XPerformanceWork

func (l PerformanceWorkItemList) ByPerformanceId(performanceId uint64) PerformanceWorkItemList {
	var list PerformanceWorkItemList
	for _, v := range l {
		if v.PerformanceId == performanceId {
			list = append(list, v)
		}
	}
	return list
}
