package report

import (
	"context"
	"database/sql"
	"errors"
	"fmt"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/tool"
	"strings"
	"time"

	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

func (m *customXReportModel) TransCtx(ctx context.Context, fn func(ctx context.Context, s sqlx.Session) error) error {
	return m.conn.TransactCtx(ctx, func(ctx context.Context, s sqlx.Session) error {
		return fn(ctx, s)
	})
}

type FindDepartmentReportsReq struct {
	EmailList    []string
	ReportType   int32
	ReportDate   time.Time
	ReportStatus int32
	IsDelayed    int32
	HasRisk      int32
}

func (m *customXReportModel) GetReportCountByUserAndType(ctx context.Context, userEmails []string, dateFrom, dateTo string) (list []*XReportCountByUserAndType, err error) {
	if len(userEmails) == 0 {
		return nil, nil
	}
	var placeholders []string
	var args []any
	for _, v := range userEmails {
		placeholders = append(placeholders, "?")
		args = append(args, v)
	}
	args = append(args, dateFrom, dateTo)
	query := fmt.Sprintf(
		`
select user_email,
       report_type,
       count(*) count
from %s
where user_email in (%s)
  and ? <= report_date
  and report_date <= ?
  and status = %d
group by user_email, report_type`,
		m.table,
		strings.Join(placeholders, ", "),
		oa.ReportStatus_DONE_RS)

	err = m.conn.QueryRowsCtx(ctx, &list, query, args...)
	return list, err
}
func (m *customXReportModel) GetReportCountByUserAndDate(ctx context.Context, userEmails []string, dateFrom, dateTo string) (list []*XReportCountByUserAndDate, err error) {
	if len(userEmails) == 0 {
		return nil, nil
	}
	var placeholders []string
	var args []any
	for _, v := range userEmails {
		placeholders = append(placeholders, "?")
		args = append(args, v)
	}
	args = append(args, dateFrom, dateTo)
	query := fmt.Sprintf(
		`
select user_email,
       report_date,
       count(*) count
from %s
where user_email in (%s)
  and ? <= report_date
  and report_date <= ?
  and status = %d
group by user_email, report_date`,
		m.table,
		strings.Join(placeholders, ", "),
		oa.ReportStatus_DONE_RS)

	err = m.conn.QueryRowsCtx(ctx, &list, query, args...)
	return list, err
}

func (m *customXReportModel) FindDepartmentReports(ctx context.Context, req FindDepartmentReportsReq) (ReportList, error) {
	if len(req.EmailList) == 0 {
		return nil, errors.New("EmailList查询条件不能为空")
	}
	var placeholders []string
	var args []any
	for _, v := range req.EmailList {
		placeholders = append(placeholders, "?")
		args = append(args, v)
	}
	args = append(args, req.ReportDate.Local().Format(time.DateOnly), req.ReportType)
	query := fmt.Sprintf("select * from %s where `user_email` in (%s) and `report_date` = ? and `report_type` = ?", m.table, strings.Join(placeholders, ", "))
	if req.ReportStatus > 0 {
		query = fmt.Sprintf("%s and `status` >= ?", query)
		args = append(args, req.ReportStatus)
	}
	if req.IsDelayed >= 0 {
		query = fmt.Sprintf("%s and `is_delayed` = ?", query)
		args = append(args, req.IsDelayed)
	}
	if req.HasRisk >= 0 {
		query = fmt.Sprintf("%s and `has_risk` = ?", query)
		args = append(args, req.HasRisk)
	}

	query = fmt.Sprintf("%s order by `status` desc", query)

	var list ReportList
	err := m.conn.QueryRowsCtx(ctx, &list, query, args...)
	return list, err
}

func (m *customXReportModel) FindReportsByUserAndReportStartDate(ctx context.Context, reportStartDate time.Time, users []string) (ReportList, error) {
	if len(users) == 0 || reportStartDate.IsZero() {
		return nil, errors.New("users, report_start_date sql查询参数必填")
	}
	placeholder := make([]string, len(users))
	args := make([]any, len(users))
	for i, v := range users {
		placeholder[i] = "?"
		args[i] = v
	}
	args = append(args, reportStartDate.Local().Format(time.DateOnly), int32(oa.ReportType_DAILY_RT), int32(oa.ReportStatus_DONE_RS))
	query := fmt.Sprintf("select * from %s where user_email in (%s) and report_date = ? and report_type = ? and `status` = ?", m.table, strings.Join(placeholder, ", "))
	var list ReportList
	err := m.conn.QueryRowsCtx(ctx, &list, query, args...)
	return list, err
}

type FindUserLatestDelayAndRiskReq struct {
	EmailList       []string
	ReportStartDate time.Time
	ReportEndDate   time.Time
}

func (m *customXReportModel) FindUserLatestDelayAndRisk(ctx context.Context, req FindUserLatestDelayAndRiskReq) (ReportList, error) {
	if len(req.EmailList) == 0 {
		return nil, nil
	}
	placeholder := make([]string, len(req.EmailList))
	args := make([]any, len(req.EmailList))
	for i, v := range req.EmailList {
		placeholder[i] = "?"
		args[i] = v
	}
	args = append(args, req.ReportStartDate, req.ReportEndDate)
	query := fmt.Sprintf(`
	select 
		r1.* 
	from %s as r1 
	inner join (
		select 
			user_email, max(report_date) as max_date 
		from %s 
		where
			user_email in (%s) and
			report_date >= ? and 
			report_date <= ? and
			status = 2
		group by user_email
	) as r2 on r1.user_email = r2.user_email and r1.report_date = r2.max_date 
	 where r1.is_delayed = 1 or r1.has_risk = 1
	`, m.table, m.table, strings.Join(placeholder, ", "))

	var list ReportList
	err := m.conn.QueryRowsCtx(ctx, &list, query, args...)
	return list, err
}

type FindUserReportsReq struct {
	UserEmail       string
	ReportStartDate time.Time
	ReportEndDate   time.Time
	ReportType      uint64
	ReportStatus    uint64
	PageNum         int64
	PageSize        int64
}

func (m *customXReportModel) FindUserReports(ctx context.Context, req FindUserReportsReq) (int64, ReportList, error) {
	countQuery := fmt.Sprintf("select count(*) from %s", m.table)
	query := fmt.Sprintf("select * from %s", m.table)
	var conditions []string
	var args []any
	if req.UserEmail != "" {
		conditions = append(conditions, "`user_email` = ?")
		args = append(args, req.UserEmail)
	}
	if !req.ReportStartDate.IsZero() {
		conditions = append(conditions, "`report_date` >= ?")
		args = append(args, req.ReportStartDate)
	}

	if !req.ReportEndDate.IsZero() {
		conditions = append(conditions, "`report_date` <= ?")
		args = append(args, req.ReportEndDate)
	}

	if req.ReportType != 0 {
		conditions = append(conditions, "`report_type` = ?")
		args = append(args, req.ReportType)
	}
	if req.ReportStatus > 0 {
		conditions = append(conditions, "`status` = ?")
		args = append(args, req.ReportStatus)
	}

	if len(conditions) > 0 {
		conditionStr := strings.Join(conditions, " and ")
		query = fmt.Sprintf("%s where %s", query, conditionStr)
		countQuery = fmt.Sprintf("%s where %s", countQuery, conditionStr)
	}

	var total int64
	err := m.conn.QueryRowCtx(ctx, &total, countQuery, args...)
	if err != nil {
		return 0, nil, err
	}
	if total == 0 {
		return 0, nil, nil
	}

	query = fmt.Sprintf("%s order by report_date desc, report_type desc", query)

	if req.PageSize > 0 && req.PageNum > 0 {
		offset := req.PageSize * (req.PageNum - 1)
		query = fmt.Sprintf("%s limit ? offset ?", query)
		args = append(args, req.PageSize, offset)
	}

	var list ReportList
	err = m.conn.QueryRowsCtx(ctx, &list, query, args...)
	if err != nil {
		return 0, nil, err
	}

	return total, list, nil
}

func (m *customXReportModel) TransInsertCtx(ctx context.Context, session sqlx.Session, data *XReport) (sql.Result, error) {
	query := fmt.Sprintf("insert into %s (%s) values (?, ?, ?, ?, ?, ?, ?, ?)", m.table, xReportRowsExpectAutoSet)
	ret, err := session.ExecCtx(ctx, query, data.UserEmail, data.ReportType, data.ReportDate, data.IsDelayed, data.DelayReason, data.HasRisk, data.RiskDesc, data.Status)
	return ret, err
}

func (m *customXReportModel) TransUpdateCtx(ctx context.Context, session sqlx.Session, newData *XReport) error {
	query := fmt.Sprintf("update %s set %s where `id` = ?", m.table, xReportRowsWithPlaceHolder)
	_, err := session.ExecCtx(ctx, query, newData.UserEmail, newData.ReportType, newData.ReportDate, newData.IsDelayed, newData.DelayReason, newData.HasRisk, newData.RiskDesc, newData.Status, newData.Id)
	return err
}

type ReportList []*XReport

func (l ReportList) Ids() []uint64 {
	ids := make([]uint64, len(l))
	for i, v := range l {
		ids[i] = v.Id
	}
	return ids
}

func (l ReportList) ByUserEmail() map[string]*XReport {
	m := make(map[string]*XReport, len(l))
	for _, v := range l {
		m[v.UserEmail] = v
	}
	return m
}

// 草稿状态、并且是自己的汇报
func (m *XReport) CanUpdate(userEmail string) bool {
	now := time.Now()
	if m.UserEmail == userEmail {
		switch m.Status {
		case uint64(oa.ReportStatus_DRAFT_RS):
			return true
		case uint64(oa.ReportStatus_DONE_RS):
			switch m.ReportType {
			case uint64(oa.ReportType_DAILY_RT):
				return tool.IsSameDay(m.UpdatedAt, now)
			case uint64(oa.ReportType_WEEKLY_RT), uint64(oa.ReportType_MONTHLY_RT):
				//周报的report_date是汇报周的周日，在下周一的0点之前都是允许修改的。
				//月报的report_date是汇报月的最后一天，在下个月1号的0点之前都是允许修改的。
				//所以周报、月报都是report_date + 1天 与当前时间对比，判断是否可修改。
				deadline := m.ReportDate.AddDate(0, 0, 1)
				return now.Before(deadline)
			}
		}
	}

	return false
}
