package report

import (
	"context"
	"database/sql"
	"fmt"
	"strings"

	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

func (m *customXReportItemModel) TransCtx(ctx context.Context, fn func(ctx context.Context, s sqlx.Session) error) error {
	return m.conn.TransactCtx(ctx, func(ctx context.Context, s sqlx.Session) error {
		return fn(ctx, s)
	})
}

func (m *customXReportItemModel) TransDeleteCtx(ctx context.Context, session sqlx.Session, id uint64) error {
	query := fmt.Sprintf("delete from %s where `id` = ?", m.table)
	_, err := session.ExecCtx(ctx, query, id)
	return err
}

func (m *customXReportItemModel) TransDeleteByReportIdCtx(ctx context.Context, session sqlx.Session, reportId uint64) error {
	query := fmt.Sprintf("delete from %s where `report_id` = ?", m.table)
	_, err := session.ExecCtx(ctx, query, reportId)
	return err
}

func (m *customXReportItemModel) TransInsertCtx(ctx context.Context, session sqlx.Session, data *XReportItem) (sql.Result, error) {
	query := fmt.Sprintf("insert into %s (%s) values (?, ?)", m.table, xReportItemRowsExpectAutoSet)
	ret, err := session.ExecCtx(ctx, query, data.ReportId, data.Content)
	return ret, err
}

func (m *customXReportItemModel) TransUpdateCtx(ctx context.Context, session sqlx.Session, data *XReportItem) error {
	query := fmt.Sprintf("update %s set %s where `id` = ?", m.table, xReportItemRowsWithPlaceHolder)
	_, err := session.ExecCtx(ctx, query, data.ReportId, data.Content, data.Id)
	return err
}

func (m *customXReportItemModel) FindByReportIds(ctx context.Context, reportIds []uint64) (ReportItemList, error) {
	if len(reportIds) == 0 {
		return nil, nil
	}
	placeholders := make([]string, len(reportIds))
	args := make([]any, len(reportIds))
	for i, v := range reportIds {
		placeholders[i] = "?"
		args[i] = v
	}

	query := fmt.Sprintf("select * from %s where `report_id` in (%s)", m.table, strings.Join(placeholders, ", "))
	var list ReportItemList
	err := m.conn.QueryRowsCtx(ctx, &list, query, args...)
	return list, err
}

func (m *customXReportItemModel) FindByIds(ctx context.Context, ids []uint64) (ReportItemList, error) {
	if len(ids) == 0 {
		return nil, nil
	}
	placeholders := make([]string, len(ids))
	args := make([]any, len(ids))
	for i, v := range ids {
		placeholders[i] = "?"
		args[i] = v
	}

	query := fmt.Sprintf("select * from %s where `id` in (%s)", m.table, strings.Join(placeholders, ", "))
	var list ReportItemList
	err := m.conn.QueryRowsCtx(ctx, &list, query, args...)
	return list, err
}

type ReportItemList []*XReportItem

func (l ReportItemList) GetByReportId(reportId uint64) ReportItemList {
	var list ReportItemList
	for _, r := range l {
		if r.ReportId == reportId {
			list = append(list, r)
		}
	}
	return list
}

func (l ReportItemList) ByReportId() map[uint64]ReportItemList {
	m := make(map[uint64]ReportItemList)
	for _, v := range l {
		if _, ok := m[v.ReportId]; !ok {
			m[v.ReportId] = ReportItemList{v}
			continue
		}
		m[v.ReportId] = append(m[v.ReportId], v)
	}

	return m
}

func (l ReportItemList) GetById(id uint64) *XReportItem {
	for _, v := range l {
		if v.Id == id {
			return v
		}
	}
	return nil
}
