package common

import (
	"fmt"
	"math"
	"oa-server/app/oacenter/oa_api/internal/types"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/format"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

var (
	ErrUsercenterUnavailable = status.Error(codes.Unavailable, "请求用户中心失败, 请求稍后再试")
)

type FormatTrend struct {
	Current any
	Last    any
	Trend   types.Trend
}

// 格式化百分比指标并计算趋势
func FormatRateMetric(current, last float64) FormatTrend {
	result := FormatTrend{
		Current: fmt.Sprintf("%.2f%%", current),
		Last:    fmt.Sprintf("%.2f%%", last),
	}
	result.Trend = calculateTrendValue(current, last)
	return result
}

// 格式化时间指标并计算趋势（秒转换为更友好的格式）
func FormatTimeMetric(current, last float64) FormatTrend {
	return FormatTrend{
		Current: format.Seconds(current),
		Last:    format.Seconds(last),
		Trend:   calculateTrendValue(current, last),
	}
}

// 计算普通指标的趋势
func CalculateTrend(current, last float64) FormatTrend {
	return FormatTrend{
		Current: current,
		Last:    last,
		Trend:   calculateTrendValue(current, last),
	}
}

// 计算趋势值和设置颜色
func calculateTrendValue(current, last float64) types.Trend {
	trend := types.Trend{}

	// 计算变化率
	var trendRate float64
	if last != 0 {
		trendRate = (current - last) / last * 100
	} else if current != 0 {
		// 如果last为0但current不为0，设置为100%增长
		trendRate = 100
	} else {
		// 如果both都为0，设置为0%变化
		trendRate = 0
	}

	// 确定趋势类型
	if trendRate > 0.1 { // 上升（允许0.1%的误差）
		trend.Type = 1 // 上升
	} else if trendRate < -0.1 { // 下降（允许0.1%的误差）
		trend.Type = 2 // 下降
	} else {
		trend.Type = 3 // 持平
	}

	// 格式化变化率
	trend.Rate = fmt.Sprintf("%.2f%%", math.Abs(trendRate))

	return trend
}

// 转换雷达图数据
func ConvertRadarChartData(rpcData *oa.RadarChartData) *types.RadarChartData {
	result := &types.RadarChartData{
		Indicator: rpcData.Indicator,
		Series:    make([]types.RadarSeries, 0, len(rpcData.Series)),
	}

	for _, series := range rpcData.Series {
		radarSeries := types.RadarSeries{
			Data: make([]types.RadarSeriesData, 0, len(series.Data)),
		}

		for _, data := range series.Data {
			radarSeries.Data = append(radarSeries.Data, types.RadarSeriesData{
				Name:  data.Name,
				Value: data.Value,
			})
		}

		result.Series = append(result.Series, radarSeries)
	}

	return result
}

// 转换折线图/柱状图数据
func ConvertLineBarChartData(rpcData *oa.LineBarChartData, typ string) *types.LineBarChartData {
	result := &types.LineBarChartData{
		XAxis:  rpcData.XAxis,
		Series: make([]types.LineBarSeriesItem, 0, len(rpcData.Series)),
	}

	for _, series := range rpcData.Series {
		result.Series = append(result.Series, types.LineBarSeriesItem{
			Name:       series.Name,
			Data:       series.Data,
			Type:       typ,
			YAxisIndex: int(series.YAxisIndex),
		})
	}

	return result
}

// 转换热力图数据
func ConvertHeatmapChartData(rpcData *oa.HeatmapChartData) *types.HeatmapChartData {
	result := &types.HeatmapChartData{
		XAxis:  rpcData.XAxis,
		YAxis:  rpcData.YAxis,
		Series: make([][]float64, 0, len(rpcData.Series)),
	}

	for _, point := range rpcData.Series {
		tmp := []float64{
			float64(point.X),
			float64(point.Y),
			point.Value,
		}

		result.Series = append(result.Series, tmp)
	}

	return result
}
