package issue

import (
	"context"
	"fmt"
	"oa-server/app/jobcenter/model"
	"oa-server/app/oacenter/oa_api/internal/svc"
	"oa-server/app/oacenter/oa_api/internal/types"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/format"
	"strconv"

	"github.com/zeromicro/go-zero/core/logx"
)

type DepartmentMetricsListLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewDepartmentMetricsListLogic(ctx context.Context, svcCtx *svc.ServiceContext) *DepartmentMetricsListLogic {
	return &DepartmentMetricsListLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

func (l *DepartmentMetricsListLogic) DepartmentMetricsList(req *types.IssueListReq) (resp *types.IssueListResp, err error) {

	columes := []types.Column{
		{
			DataIndex: "nickname",
			Title:     "姓名",
		},
		{
			DataIndex: "department",
			Title:     "部门",
		},
		{
			DataIndex: "prodBugCount",
			Title:     "线上BUG数",
		},
		{
			DataIndex: "nonProdBugCount",
			Title:     "线下BUG数",
		},
		{
			DataIndex: "score",
			Title:     "得分",
		},
	}
	expandedColumns := []types.Column{
		{
			DataIndex: "type",
			Title:     "类型",
		},
		{
			DataIndex: "reason",
			Title:     "原因",
		},
		{
			DataIndex: "priority",
			Title:     "优先级",
		},
		{
			DataIndex: "count",
			Title:     "数量",
		},
	}

	rpcResp, err := l.svcCtx.IssueRpc.DepartmentIssueList(l.ctx, &oa.IssueListReq{
		PeriodType:    oa.PeriodType(req.PeriodType),
		DepartmentKey: req.DepartmentKey,
		DateFromTo:    req.DateFromTo,
		PageNum:       int64(req.Page),
		PageSize:      int64(req.PageSize),
		Employee:      &req.Employee,
	})
	if err != nil {
		logx.Errorw("IssueListLogic IssueList error", logx.Field("error", err), logx.Field("department", req.DepartmentKey))
		return &types.IssueListResp{
			Columns:         columes,
			ExpandedColumns: expandedColumns,
		}, err
	}
	if len(rpcResp.List) == 0 {
		return &types.IssueListResp{
			Columns:         columes,
			ExpandedColumns: expandedColumns,
		}, nil
	}
	//  从rpcResp中获取其中一个员工的角色
	userInfo, err := l.svcCtx.UserCache.GetUserByEmail(l.ctx, rpcResp.List[0].Employee)
	if err != nil {
		logx.Errorw("IssueListLogic DepartmentMetricsList error", logx.Field("error", err), logx.Field("department", req.DepartmentKey))
		return &types.IssueListResp{
			Columns:         columes,
			ExpandedColumns: expandedColumns,
		}, err
	}
	role := userInfo.UserRole
	if role == model.RoleTester {
		lastColumn := columes[len(columes)-1]
		columes = columes[:len(columes)-1]

		columes = append(columes, types.Column{
			DataIndex: "bugEscapeRate",
			Title:     "BUG逃逸率",
		})

		columes = append(columes, lastColumn)
	}
	data := make([]types.Data, 0)
	for _, v := range rpcResp.List {
		tmp := types.Data{
			ID:              strconv.FormatInt(v.Id, 10),
			Department:      v.DepartmentName,
			NickName:        v.Nickname,
			Email:           v.Employee,
			ProdBugCount:    int(v.ProdBugCount),
			NonProdBugCount: int(v.NonProdBugCount),
			Score:           format.FormatFloatWithPrecision(float64(v.Score), 2),
		}
		if role == model.RoleTester {
			tmp.BugEscapeRate = fmt.Sprintf("%.2f%%", v.BugEscapeRate)
		}
		for _, info := range v.Infos {
			tmp.ExpandedData = append(tmp.ExpandedData, types.ExpandedData{
				Type:     info.Type,
				Count:    int(info.Count),
				Reason:   info.Reason,
				Priority: info.Priority,
			})
		}

		data = append(data, tmp)
	}
	return &types.IssueListResp{
		Columns:         columes,
		ExpandedColumns: expandedColumns,
		Data:            data,
		Total:           rpcResp.Total,
	}, nil
}
