package issue

import (
	"context"
	"fmt"

	"oa-server/app/jobcenter/model"
	"oa-server/app/oacenter/oa_api/internal/logic/common"
	"oa-server/app/oacenter/oa_api/internal/svc"
	"oa-server/app/oacenter/oa_api/internal/types"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type PersonalMetricsLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewPersonalMetricsLogic(ctx context.Context, svcCtx *svc.ServiceContext) *PersonalMetricsLogic {
	return &PersonalMetricsLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

func (l *PersonalMetricsLogic) PersonalMetrics(req *types.IssuePersonalMetricsReq) (resp *types.IssuePersonalMetricsResp, err error) {

	rpcReq := &oa.IssuePersonalMetricsReq{
		PeriodType: oa.PeriodType(req.PeriodType),
		DateFromTo: req.DateFromTo,
		Employee:   &req.Employee,
	}
	rpcResp, err := l.svcCtx.IssueRpc.PersonalIssueMetrics(l.ctx, rpcReq)
	if err != nil {
		logx.Errorw("获取个人issue指标失败", logx.Field("error", err), logx.Field("employee", req.Employee))
		resp = getMetricInfo()
		resp.Nickname = req.Employee
		return resp, nil
	}
	userInfo, err := l.svcCtx.UserCache.GetUserByEmail(l.ctx, req.Employee)
	if err != nil {
		logx.Errorw("获取个人issue指标失败", logx.Field("error", err), logx.Field("employee", req.Employee))
		return nil, fmt.Errorf("获取个人issue指标失败: %v", err)
	}
	role := userInfo.UserRole

	resp = getMetricInfo()

	prodBugMetric := common.FormatRateMetric(rpcResp.ProdBug.Current, rpcResp.ProdBug.Last)
	nonProdBugMetric := common.FormatRateMetric(rpcResp.NonProdBug.Current, rpcResp.NonProdBug.Last)
	bugEscapeMetric := common.FormatTrend{}
	if role == model.RoleTester {
		bugEscapeMetric = common.FormatRateMetric(rpcResp.BugEscapeRate.Current, rpcResp.BugEscapeRate.Last)
	}

	resp.Id = rpcResp.Id
	resp.Nickname = rpcResp.NickName
	resp.ProdBug.Current = rpcResp.ProdBug.Current
	resp.ProdBug.Last = rpcResp.ProdBug.Last
	resp.ProdBug.Score = rpcResp.ProdBug.Score
	resp.ProdBug.Trend = types.Trend{
		Type: prodBugMetric.Trend.Type,
		Rate: prodBugMetric.Trend.Rate,
	}
	resp.NonProdBug.Current = rpcResp.NonProdBug.Current
	resp.NonProdBug.Last = rpcResp.NonProdBug.Last
	resp.NonProdBug.Score = rpcResp.NonProdBug.Score
	resp.NonProdBug.Trend = types.Trend{
		Type: nonProdBugMetric.Trend.Type,
		Rate: nonProdBugMetric.Trend.Rate,
	}
	if role == model.RoleTester {
		resp.BugEscapeRate.Current = rpcResp.BugEscapeRate.Current
		resp.BugEscapeRate.Last = rpcResp.BugEscapeRate.Last
		resp.BugEscapeRate.Score = rpcResp.BugEscapeRate.Score
		resp.BugEscapeRate.Trend = types.Trend{
			Type: bugEscapeMetric.Trend.Type,
			Rate: bugEscapeMetric.Trend.Rate,
		}
	}
	resp.ScoreInfo.Score = rpcResp.Score
	resp.ScoreInfo.Rank = rpcResp.Rank
	resp.ScoreInfo.Name = "BUG统计绩效评分"
	resp.ScoreInfo.Rank = rpcResp.Rank
	return resp, nil

}

// getMetricInfo 获取指标文案信息
func getMetricInfo() *types.IssuePersonalMetricsResp {
	resp := &types.IssuePersonalMetricsResp{
		ProdBug: types.MetricItemData{

			MetricInfo: types.MetricInfo{
				Name:        "生产BUG数",
				Tips:        "生产BUG总数-误报数",
				CalcFormula: "评分规则：\n=0，+100分\n>0且≤2，+60分\n>2且≤5，+30分\n>5，+10分",
			},
		},
		NonProdBug: types.MetricItemData{
			MetricInfo: types.MetricInfo{
				Name:        "非生产BUG数",
				Tips:        "非生产BUG总数-误报数",
				CalcFormula: "评分规则：\n=0，+100分\n>0且≤5，+80分\n>5且≤10，+60分\n>10，+30分",
			},
		},
		BugEscapeRate: types.MetricItemData{
			MetricInfo: types.MetricInfo{
				Name:        "BUG逃逸率",
				Tips:        "生产BUG数/(生产BUG数+非生产BUG数)，测试人员指标，非测试人员不参考此项打分",
				CalcFormula: "评分规则：\n=0%，+100分\n>0%且≤5%，+80分\n>5%且≤10%，+60分\n>10%，+10分",
			},
		},
		ScoreInfo: types.ScoreInfo{
			Tips:        "本周期BUG绩效评分",
			CalcFormula: "计算公式请在人效核算界面查询",
			Name:        "BUG绩效评分",
		},
	}

	return resp
}
