package monitor

import (
	"context"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/format"

	"oa-server/app/oacenter/oa_api/internal/svc"
	"oa-server/app/oacenter/oa_api/internal/types"

	"github.com/zeromicro/go-zero/core/logx"
)

type MonitorMetricsListLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewMonitorMetricsListLogic(ctx context.Context, svcCtx *svc.ServiceContext) *MonitorMetricsListLogic {
	return &MonitorMetricsListLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

type MetricsList struct {
	Id                     int64   `json:"id"`
	Nickname               string  `json:"nickname"`
	DepartmentName         string  `json:"department_name"`
	EffectiveHandledEvents int64   `json:"effective_handled_events"`
	AverageProcessingTime  string  `json:"average_processing_time"`
	AverageResolutionTime  string  `json:"average_resolution_time"`
	FailureRecurrenceRate  string  `json:"failure_recurrence_rate"`
	BizEventAssignmentRate string  `json:"biz_event_assignment_rate"`
	BizRuleHitRate         string  `json:"biz_rule_hit_rate"`
	MonitorScore           float64 `json:"monitor_score"`
	Email                  string  `json:"email"`
}

func (l *MonitorMetricsListLogic) MonitorMetricsList(req *types.MonitorMetricsListReq) (resp *types.MonitorMetricsListResp, err error) {
	rpcResp, err := l.svcCtx.MonitorRpc.GetMonitorMetricsList(l.ctx, &oa.MetricsListReq{
		Page:          int32(req.Page),
		PageSize:      int32(req.PageSize),
		DepartmentKey: req.DepartmentKey,
		DateFromTo:    req.DateFromTo,
		Employee:      &req.Employee,
		PeriodType:    oa.PeriodType(req.PeriodType),
	})
	if err != nil {
		return nil, err
	}
	list := []MetricsList{}

	if len(rpcResp.List) == 0 {
		return nil, nil
	}
	for _, metrics := range rpcResp.List {
		tmp := MetricsList{
			Id:                     metrics.Id,
			Nickname:               metrics.Nickname,
			DepartmentName:         metrics.DepartmentName,
			EffectiveHandledEvents: metrics.EffectiveHandledEvents,
			AverageProcessingTime:  format.Seconds(float64(metrics.AverageProcessingTime)),
			AverageResolutionTime:  format.Seconds(float64(metrics.AverageResolutionTime)),
			FailureRecurrenceRate:  format.FormatFloatWithPrecision(float64(metrics.FailureRecurrenceRate), 2) + "%",
			BizEventAssignmentRate: format.FormatFloatWithPrecision(float64(metrics.BizEventAssignmentRate), 2) + "%",
			BizRuleHitRate:         format.FormatFloatWithPrecision(float64(metrics.BizRuleHitRate), 2) + "%",
			MonitorScore:           metrics.MonitorScore,
			Email:                  metrics.Email,
		}
		list = append(list, tmp)
	}
	return &types.MonitorMetricsListResp{
		List:  list,
		Total: int(rpcResp.Total),
	}, nil
}
