package monitor

import (
	"context"

	"oa-server/app/oacenter/oa_api/internal/logic/common"
	"oa-server/app/oacenter/oa_api/internal/svc"
	"oa-server/app/oacenter/oa_api/internal/types"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type PersonalMetricsLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewPersonalMetricsLogic(ctx context.Context, svcCtx *svc.ServiceContext) *PersonalMetricsLogic {
	return &PersonalMetricsLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

// getMetricInfo 获取指标文案信息
func getMetricInfo() *types.PersonalMetricsResp {
	return &types.PersonalMetricsResp{
		FailureRecurrenceRate: types.MetricItemData{
			MetricInfo: types.MetricInfo{
				Name:        "故障复发率",
				Tips:        "激活总数/事件总数",
				CalcFormula: "评分规则：\n≤5%，+60分\n≤10%，+30分",
			},
		},
		AverageProcessingTime: types.MetricItemData{
			MetricInfo: types.MetricInfo{
				Name:        "平均响应时间",
				Tips:        "响应总时间(从分配到开始处理时间)/分配事件总数",
				CalcFormula: "评分规则：\n≤20分钟，+100分\n≤60分钟，+80分\n≤120分钟，+60分\n≤240分钟，+30分\n≤600分钟，+10分",
			},
		},
		AverageResolutionTime: types.MetricItemData{
			MetricInfo: types.MetricInfo{
				Name:        "平均处理时间",
				Tips:        "处理完成总时间（从开始处理到处理完成时间）/处理完成总数",
				CalcFormula: "评分规则：\n≤4h，+100分\n≤12h，+80分\n≤24h，+60分\n≤72h，+40分\n≤120h，+20分\n>120h，+10分",
			},
		},
		EffectiveHandledEvents: types.MetricItemData{
			MetricInfo: types.MetricInfo{
				Name:        "有效处理事件",
				Tips:        "总完成事件数-误报事件-激活事件-挂起事件",
				CalcFormula: "评分规则：\n≥阈值，+100分\n≥阈值80%，+90分\n≥阈值60%，+80分\n≥阈值40%，+70分\n≥阈值20%，+50分\n≥阈值10%，+30分\n(阈值：周10，月40，双月80，季度120)",
			},
		},
		BizEventAssignmentRate: types.MetricItemData{
			MetricInfo: types.MetricInfo{
				Name:        "事件分配率",
				Tips:        "已指派处理人事件数/部门事件数。部门指标，非leader员工不参考此项打分。",
				CalcFormula: "评分规则：\n≥60%，+100分\n≥30%，+70分\n≥10%，+30分",
			},
		},
		BizRuleHitRate: types.MetricItemData{
			MetricInfo: types.MetricInfo{
				Name:        "规则命中率",
				Tips:        "部门事件数/源数据总数。部门指标，非leader员工不参考此项打分。",
				CalcFormula: "评分规则：\n≥10%，+100分\n≥8%，+90分\n≥6%，+80分\n≥4%，+70分\n≥2%，+50分\n≥1%，+30分\n≥0%，+10分",
			},
		},
		ScoreInfo: types.ScoreInfo{
			Tips:        "本周期服务治理个人得分",
			CalcFormula: "计算公式请在人效核算界面查询",
			Name:        "服务治理绩效评分",
		},
	}
}

func (l *PersonalMetricsLogic) PersonalMetrics(req *types.PersonalMetricsReq) (resp *types.PersonalMetricsResp, err error) {
	rpcReq := &oa.PersonalMetricsReq{
		PeriodType: oa.PeriodType(req.PeriodType),
		DateFromTo: req.DateFromTo,
		Employee:   &req.Employee,
	}
	rpcResp, err := l.svcCtx.MonitorRpc.GetPersonalMetrics(l.ctx, rpcReq)
	if err != nil {
		l.Logger.Error("获取个人指标数据失败，只返回文案提示", logx.Field("error", err), logx.Field("employee", req.Employee))
		resp = getMetricInfo()
		resp.Nickname = req.Employee
		return resp, nil
	}

	failureRateMetric := common.FormatRateMetric(rpcResp.FailureRecurrenceRate.Current, rpcResp.FailureRecurrenceRate.Last)
	avgProcTimeMetric := common.FormatTimeMetric(rpcResp.AverageProcessingTime.Current, rpcResp.AverageProcessingTime.Last)
	avgResolTimeMetric := common.FormatTimeMetric(rpcResp.AverageResolutionTime.Current, rpcResp.AverageResolutionTime.Last)
	effectiveEventsMetric := common.CalculateTrend(rpcResp.EffectiveHandledEvents.Current, rpcResp.EffectiveHandledEvents.Last)
	bizAssignRateMetric := common.FormatRateMetric(rpcResp.BizEventAssignmentRate.Current, rpcResp.BizEventAssignmentRate.Last)
	bizRuleRateMetric := common.FormatRateMetric(rpcResp.BizRuleHitRate.Current, rpcResp.BizRuleHitRate.Last)

	resp = getMetricInfo()
	resp.Id = rpcResp.Id
	resp.Nickname = rpcResp.Nickname
	resp.FailureRecurrenceRate.Current = failureRateMetric.Current
	resp.FailureRecurrenceRate.Last = failureRateMetric.Last
	resp.FailureRecurrenceRate.Score = rpcResp.FailureRecurrenceRate.Score
	resp.FailureRecurrenceRate.Trend = failureRateMetric.Trend

	resp.AverageProcessingTime.Current = avgProcTimeMetric.Current
	resp.AverageProcessingTime.Last = avgProcTimeMetric.Last
	resp.AverageProcessingTime.Score = rpcResp.AverageProcessingTime.Score
	resp.AverageProcessingTime.Trend = avgProcTimeMetric.Trend

	resp.AverageResolutionTime.Current = avgResolTimeMetric.Current
	resp.AverageResolutionTime.Last = avgResolTimeMetric.Last
	resp.AverageResolutionTime.Score = rpcResp.AverageResolutionTime.Score
	resp.AverageResolutionTime.Trend = avgResolTimeMetric.Trend

	resp.EffectiveHandledEvents.Current = effectiveEventsMetric.Current
	resp.EffectiveHandledEvents.Last = effectiveEventsMetric.Last
	resp.EffectiveHandledEvents.Score = rpcResp.EffectiveHandledEvents.Score
	resp.EffectiveHandledEvents.Trend = effectiveEventsMetric.Trend

	resp.BizEventAssignmentRate.Current = bizAssignRateMetric.Current
	resp.BizEventAssignmentRate.Last = bizAssignRateMetric.Last
	resp.BizEventAssignmentRate.Score = rpcResp.BizEventAssignmentRate.Score
	resp.BizEventAssignmentRate.Trend = bizAssignRateMetric.Trend

	resp.BizRuleHitRate.Current = bizRuleRateMetric.Current
	resp.BizRuleHitRate.Last = bizRuleRateMetric.Last
	resp.BizRuleHitRate.Score = rpcResp.BizRuleHitRate.Score
	resp.BizRuleHitRate.Trend = bizRuleRateMetric.Trend

	resp.ScoreInfo.Score = rpcResp.Score
	resp.ScoreInfo.Rank = rpcResp.Rank

	return resp, nil
}
