package performance

import (
	"context"
	"time"

	"oa-server/app/oacenter/oa_api/internal/svc"
	"oa-server/app/oacenter/oa_api/internal/types"
	"oa-server/app/oacenter/oa_rpc/client/performancerpc"
	"oa-server/common/ctxdata"

	"github.com/zeromicro/go-zero/core/logx"
)

type GetPerformanceDetailLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewGetPerformanceDetailLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetPerformanceDetailLogic {
	return &GetPerformanceDetailLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

func (l *GetPerformanceDetailLogic) GetPerformanceDetail(req *types.GetPerformanceDetailReq) (resp *types.GetPerformanceDetailResp, err error) {
	userEmail := l.ctx.Value(ctxdata.CtxKeyJwtUserEmail).(string)
	in := &performancerpc.GetPerformanceDetailReq{PerformanceId: req.PerformanceID, UserEmail: userEmail}
	out, err := l.svcCtx.PerformanceRpc.GetPerformanceDetail(l.ctx, in)
	if err != nil {
		l.Logger.Errorw("failed to call performanceRpc.GetPerformanceDetail", logx.Field("id", req.PerformanceID), logx.Field("error", err))
		return
	}
	workItems := make([]types.PerformanceWorkItem, len(out.GetWorkItems()))
	for i, v := range out.GetWorkItems() {
		comments := make([]types.PerformanceItemComment, len(v.GetComments()))
		for j, m := range v.GetComments() {
			comments[j] = types.PerformanceItemComment{
				CommentID:   m.GetCommentId(),
				ItemID:      m.GetItemId(),
				CommentType: uint64(m.GetCommentType()),
				Comment:     m.GetComment(),
				UserEmail:   m.GetUserEmail(),
				CreatedAt:   m.GetCreatedAt().AsTime().Local().Format(time.DateTime),
				UpdatedAt:   m.GetUpdatedAt().AsTime().Local().Format(time.DateTime),
			}
		}
		workItems[i] = types.PerformanceWorkItem{
			ID:            v.Id,
			ItemName:      v.ItemName,
			ItemWeight:    v.ItemWeight,
			ItemDesc:      v.ItemDesc,
			ItemDeficency: v.ItemDeficency,
			SelfScore:     v.SelfScore,
			LeaderScore:   v.LeaderScore,
			Comments:      comments,
		}
	}
	additionItems := make([]types.PerformanceAdditionItem, len(out.GetAdditionItems()))
	for i, v := range out.GetAdditionItems() {
		comments := make([]types.PerformanceItemComment, len(v.GetComments()))
		for j, m := range v.GetComments() {
			comments[j] = types.PerformanceItemComment{
				CommentID:   m.GetCommentId(),
				ItemID:      m.GetItemId(),
				CommentType: uint64(m.GetCommentType()),
				UserEmail:   m.GetUserEmail(),
				Comment:     m.GetComment(),
				CreatedAt:   m.GetCreatedAt().AsTime().Local().Format(time.DateTime),
				UpdatedAt:   m.GetUpdatedAt().AsTime().Local().Format(time.DateTime),
			}
		}
		additionItems[i] = types.PerformanceAdditionItem{
			ID:            v.Id,
			ItemName:      v.ItemName,
			ItemType:      int32(v.ItemType),
			ItemWeight:    v.ItemWeight,
			ItemDesc:      v.ItemDesc,
			SelfScore:     v.SelfScore,
			LeaderScore:   v.LeaderScore,
			CoworkerEmail: v.CoworkerEmail,
			Comments:      comments,
		}
	}

	p := out.GetPerformance()
	resp = &types.GetPerformanceDetailResp{
		Performance: types.Performance{
			ID:                 p.GetId(),
			UserEmail:          p.GetUserEmail(),
			Year:               p.GetYear(),
			MonthNum:           p.GetMonthNum(),
			WorkItemWeight:     p.GetWorkItemWeight(),
			AdditionItemWeight: p.GetAdditionItemWeight(),
			TotalScore:         p.GetTotalScore(),
			Level:              p.GetLevel(),
			LeaderWords:        p.GetLeaderWords(),
			Status:             int32(p.GetStatus()),
			Department: types.Department{
				DepartmentKey:  p.GetDepartment().GetKey(),
				DepartmentName: p.GetDepartment().GetName(),
			},
			CreatedAt: p.GetCreatedAt().AsTime().Local().Format(time.DateTime),
			UpdatedAt: p.GetUpdatedAt().AsTime().Local().Format(time.DateTime),
		},
		WorkItems:     workItems,
		AdditionItems: additionItems,
	}

	return
}
