package performance

import (
	"context"
	"sort"
	"time"

	"oa-server/app/oacenter/oa_api/internal/svc"
	"oa-server/app/oacenter/oa_api/internal/types"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/ctxdata"

	"github.com/zeromicro/go-zero/core/logx"
)

type GetPerformanceItemCommentListLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewGetPerformanceItemCommentListLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetPerformanceItemCommentListLogic {
	return &GetPerformanceItemCommentListLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

func (l *GetPerformanceItemCommentListLogic) GetPerformanceItemCommentList(req *types.GetPerformanceItemCommentListReq) (resp *types.GetPerformanceItemCommentListResp, err error) {
	userEmail := l.ctx.Value(ctxdata.CtxKeyJwtUserEmail).(string)
	in := &oa.GetPerformanceItemCommentListReq{
		PerformanceId: req.PerformanceID,
		ItemId:        req.ItemID,
		CommentType:   oa.PerformanceCommentType(req.CommentType),
		UserEmail:     userEmail,
		PageNum:       1,
		PageSize:      5,
		OrderBy:       "`id` desc",
	}
	out, err := l.svcCtx.PerformanceRpc.GetPerformanceItemCommentList(l.ctx, in)
	if err != nil {
		l.Logger.Errorw("failed to call performanceRpc.GetPerformanceItemCommentList", logx.Field("error", err))
		return nil, err
	}

	comments := make([]types.PerformanceItemComment, out.Total)
	for i, v := range out.List {
		comments[i] = types.PerformanceItemComment{
			CommentID:   v.GetCommentId(),
			CommentType: uint64(v.GetCommentType()),
			UserEmail:   v.GetUserEmail(),
			Comment:     v.GetComment(),
			ItemID:      v.GetItemId(),
			CreatedAt:   v.CreatedAt.AsTime().Local().Format(time.DateTime),
			UpdatedAt:   v.UpdatedAt.AsTime().Local().Format(time.DateTime),
		}
	}

	// 由于请求的时候是按id倒序排序的，所以还需要恢复评论的顺序。
	sort.SliceStable(comments, func(i, j int) bool { return comments[i].CommentID < comments[j].CommentID })

	return &types.GetPerformanceItemCommentListResp{Total: out.Total, List: comments}, nil
}
