package performance

import (
	"context"
	"time"

	"oa-server/app/oacenter/oa_api/internal/svc"
	"oa-server/app/oacenter/oa_api/internal/types"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/ctxdata"

	"github.com/zeromicro/go-zero/core/logx"
)

type GetPerformanceRankingLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewGetPerformanceRankingLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetPerformanceRankingLogic {
	return &GetPerformanceRankingLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

func (l *GetPerformanceRankingLogic) GetPerformanceRanking(req *types.GetPerformanceRankingReq) (resp *types.GetPerformanceRankingResp, err error) {
	userEmail := l.ctx.Value(ctxdata.CtxKeyJwtUserEmail).(string)
	in := &oa.GetPerformanceRankingReq{
		UserEmail:           userEmail,
		Year:                req.Year,
		MonthNum:            req.MonthNum,
		SortField:           req.SortField,
		SortOrder:           req.SortOrder,
		PageNum:             req.PageNum,
		PageSize:            req.PageSize,
		SearchUserEmail:     req.UserEmail,
		DepartmentKey:       req.DepartmentKey,
		IsExcellentEmployee: req.IsExcellentEmployee,
		Status:              req.Status,
	}
	out, err := l.svcCtx.PerformanceRpc.GetPerformanceRanking(l.ctx, in)
	if err != nil {
		l.Logger.Errorw("failed to call performanceRpc.GetPerformanceRanking", logx.Field("error", err))
		return nil, err
	}
	statistics := make([]types.Statistics, len(out.GetStatistics()))
	for i, s := range out.GetStatistics() {
		statistics[i] = types.Statistics{
			Level:   s.Level,
			Count:   s.Count,
			Percent: s.Percent,
		}
	}
	list := make([]types.PerformanceRanking, len(out.GetList()))
	for i, p := range out.GetList() {
		wList := make([]types.PerformanceWorkItem, len(p.WorkItems))
		for j, w := range p.WorkItems {
			wList[j] = types.PerformanceWorkItem{
				ID:            w.Id,
				ItemName:      w.ItemName,
				ItemWeight:    w.ItemWeight,
				ItemDesc:      w.ItemDesc,
				ItemDeficency: w.ItemDeficency,
				SelfScore:     w.SelfScore,
				LeaderScore:   w.LeaderScore,
			}
		}
		list[i] = types.PerformanceRanking{
			PerformanceID:           p.PerformanceId,
			Year:                    p.Year,
			MonthNum:                p.MonthNum,
			UserEmail:               p.UserEmail,
			Email:                   p.Email,
			TotalScore:              p.TotalScore,
			TotalBonus:              p.TotalBonus,
			TimeFactor:              p.TimeFactor,
			Level:                   p.Level,
			LevelFactor:             p.LevelFactor,
			IncidentDeduction:       p.IncidentDeduction,
			ExcellentMemberBonus:    p.ExcellentMemberBonus,
			ExcellentManagerBonus:   p.ExcellentManagerBonus,
			IsExcellentEmployee:     p.IsExcellentEmployee,
			ExcellentEmployeeReason: p.ExcellentEmployeeReason,
			Status:                  int32(p.Status),
			CreatedAt:               p.CreatedAt.AsTime().Local().Format(time.DateTime),
			UpdatedAt:               p.UpdatedAt.AsTime().Local().Format(time.DateTime),
			WorkItems:               wList,
		}
	}

	resp = &types.GetPerformanceRankingResp{
		Total:      out.GetTotal(),
		Statistics: statistics,
		TotalBonus: out.GetTotalBonus(),
		List:       list,
	}
	return
}
