package calcrpclogic

import (
	"context"

	"oa-server/app/oacenter/model/efficiency_calc"
	"oa-server/app/oacenter/oa_rpc/internal/logic/tools"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type CalcListLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewCalcListLogic(ctx context.Context, svcCtx *svc.ServiceContext) *CalcListLogic {
	return &CalcListLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *CalcListLogic) CalcList(in *oa.CalcListReq) (*oa.CalcListResp, error) {
	//  考虑到用户不会有多个一级部门，当department_key为空时，查询所有二级部门
	//  如果parent_department_key不为空，还是查询所有二级部门
	departmentsKey := []string{in.DepartmentKey}
	if in.DepartmentKey == "" {
		userInfo, err := l.svcCtx.UserCache.GetUserByEmail(l.ctx, in.Employee)
		if err != nil {
			logx.Errorw("GetUserByEmail failed", logx.Field("error", err))
			return nil, err
		}
		for _, v := range userInfo.SecondDepartmentList {
			departmentsKey = append(departmentsKey, v.DepartmentKey)
		}
	}

	total, list, err := l.svcCtx.EfficiencyCalcModel.List(l.ctx, &efficiency_calc.ListReq{
		Page:             int(in.PageNum),
		PageSize:         int(in.PageSize),
		Role:             oa.Role(in.Role),
		DepartmentKey:    departmentsKey,
		ParentDepartment: in.ParentDepartmentKey,
		Index:            oa.CalcIndex(in.Index),
	})
	if err != nil {
		logx.Errorw("List failed", logx.Field("error", err))
		return nil, err
	}
	departmentMap, err := l.svcCtx.UserCache.GetDepartmentMap(l.ctx)
	if err != nil {
		logx.Errorw("GetDepartmentMap failed", logx.Field("error", err))
		return nil, err
	}
	respList := make([]*oa.CalcList, 0)
	for _, v := range list {
		respList = append(respList, &oa.CalcList{
			Id:                   int64(v.Id),
			DepartmentKey:        v.DepartmentKey.String,
			DepartmentName:       departmentMap[v.DepartmentKey.String].DepartmentName,
			ParentDepartmentKey:  v.ParentDepartmentKey.String,
			ParentDepartmentName: departmentMap[v.ParentDepartmentKey.String].DepartmentName,
			Role:                 oa.Role(v.Role.Int64),
			Index:                oa.CalcIndex(v.CalcIndex.Int64),
			EffectiveTime:        v.EffectiveTime.Time.Format("2006-01"),
			UpdateTime:           v.UpdatedAt.Format("2006-01-02 15:04:05"),
			UpdateUser:           v.UpdateBy.String,
			CalcFormula:          v.CalcFormula,
			IndexName:            tools.CalcIndexMap[v.CalcIndex.Int64],
			RoleName:             tools.RoleMap[v.Role.Int64],
		})
	}

	return &oa.CalcListResp{
		Total: int64(total),
		List:  respList,
	}, nil
}
