package calcrpclogic

import (
	"context"
	"database/sql"
	"encoding/json"
	"time"

	"oa-server/app/jobcenter/model"
	"oa-server/app/oacenter/model/efficiency_calc"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/constant"

	"github.com/spf13/cast"
	"github.com/zeromicro/go-zero/core/logx"
)

type DepartmentModifyLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewDepartmentModifyLogic(ctx context.Context, svcCtx *svc.ServiceContext) *DepartmentModifyLogic {
	return &DepartmentModifyLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 服务治理开发公式
var DeveloperMonitorScoreFormula = []*oa.Item{
	{
		Key:   constant.FailureRecurrenceRate,
		Value: "0.3",
	},
	{
		Key:   constant.AverageResponseTime,
		Value: "0.3",
	},
	{
		Key:   constant.AverageProcessingTime,
		Value: "0.2",
	},
	{
		Key:   constant.EffectiveHandledEvents,
		Value: "0.1",
	},
	{
		Key:   constant.BizEventAssignmentRate,
		Value: "0.05",
	},
	{
		Key:   constant.BizRuleHitRate,
		Value: "0.05",
	},
}

// 交付质量开发公式
var DeveloperDeliveryQualityFormula = []*oa.Item{
	{
		Key:   constant.ProdBugCount,
		Value: "0.7",
	},
	{
		Key:   constant.NonProdBugCount,
		Value: "0.3",
	},
	{
		Key:   constant.BugEscapeRate,
		Value: "0",
	},
}

// 交付质量测试公式
var TesterDeliveryQualityFormula = []*oa.Item{
	{
		Key:   constant.ProdBugCount,
		Value: "0.3",
	},
	{
		Key:   constant.NonProdBugCount,
		Value: "0.3",
	},
	{
		Key:   constant.BugEscapeRate,
		Value: "0.4",
	},
}

// DepartmentModify 部门变动，需要为新增部门初始化计算公式
func (l *DepartmentModifyLogic) DepartmentModify(in *oa.DepartmentModifyReq) (*oa.EmptyResponse, error) {
	departmentMap, err := l.svcCtx.UserCache.GetDepartmentMap(l.ctx)
	if err != nil {
		logx.Errorw("GetDepartmentMap failed", logx.Field("error", err))
		return nil, err
	}

	for _, v := range in.DepartmentInfo {
		if cast.ToInt(v.Value) <= 1 {
			// 部门未设置角色，不设置公式
			continue
		}
		//  只配置二级部门
		if departmentMap[v.Key].ParentDepartmentKey != "od-8962594c3fa24ef77cb05033c6d4f903" {
			continue
		}

		if cast.ToInt(v.Value) == int(model.RoleDeveloper) {
			// 服务治理开发公式
			err := l.insertFormulas(v.Key, departmentMap, model.RoleDeveloper, DeveloperMonitorScoreFormula, oa.CalcIndex_MONITOR_SCORE)
			if err != nil {
				logx.Errorw("Insert failed", logx.Field("error", err))
				return nil, err
			}
			// 交付质量开发公式
			err = l.insertFormulas(v.Key, departmentMap, model.RoleDeveloper, DeveloperDeliveryQualityFormula, oa.CalcIndex_DELIVERY_QUALITY)
			if err != nil {
				logx.Errorw("Insert failed", logx.Field("error", err))
				return nil, err
			}
		}
		if cast.ToInt(v.Value) == int(model.RoleTester) {
			// 交付质量测试公式
			err := l.insertFormulas(v.Key, departmentMap, model.RoleTester, TesterDeliveryQualityFormula, oa.CalcIndex_DELIVERY_QUALITY)
			if err != nil {
				logx.Errorw("Insert failed", logx.Field("error", err))
				return nil, err
			}
		}
	}

	return &oa.EmptyResponse{}, nil
}

// insertFormulas 插入指定角色下所有需要的公式
func (l *DepartmentModifyLogic) insertFormulas(
	departmentKey string,
	departmentMap map[string]model.DepartmentInfo,
	role model.UserRole,
	calcFormula []*oa.Item,
	index oa.CalcIndex,
) error {
	// 查询是否存在，如果存在就跳过
	total, _, err := l.svcCtx.EfficiencyCalcModel.List(l.ctx, &efficiency_calc.ListReq{
		DepartmentKey: []string{departmentKey},
		Role:          oa.Role(role),
		Index:         index,
	})
	if err != nil {
		logx.Errorw("FindOneByDepartmentKeyAndRoleAndIndex failed", logx.Field("error", err))
		return err
	}
	if total > 0 {
		return nil
	}
	// 生效时间，当年1月，使用东八区时间
	effectiveTime := time.Date(time.Now().Year(), 1, 1, 0, 0, 0, 0, time.FixedZone("CST", 8*3600))
	logx.Infow("effectiveTime", logx.Field("effectiveTime", effectiveTime))
	jsonFormula, err := json.Marshal(calcFormula)
	if err != nil {
		logx.Errorw("Marshal failed", logx.Field("error", err))
		return err
	}
	result, err := l.svcCtx.EfficiencyCalcModel.Insert(l.ctx, &efficiency_calc.XEfficiencyCalc{
		DepartmentKey:       sql.NullString{String: departmentKey, Valid: true},
		ParentDepartmentKey: sql.NullString{String: departmentMap[departmentKey].ParentDepartmentKey, Valid: true},
		Role:                sql.NullInt64{Int64: int64(role), Valid: true},
		CalcIndex:           sql.NullInt64{Int64: int64(index), Valid: true},
		CalcFormula:         sql.NullString{String: string(jsonFormula), Valid: true},
		EffectiveTime:       sql.NullTime{Time: effectiveTime, Valid: true},
		UpdateBy:            sql.NullString{String: "system", Valid: true},
	})
	if err != nil {
		logx.Errorw("Insert failed", logx.Field("error", err))
		return err
	}
	lastInsertId, err := result.LastInsertId()
	if err != nil {
		logx.Errorw("get LastInsertId failed", logx.Field("error", err))
		return err
	}
	_, err = l.svcCtx.EfficiencyCalcHistoryModel.Insert(l.ctx, &efficiency_calc.XEfficiencyCalcHistory{
		DepartmentKey: sql.NullString{String: departmentKey, Valid: true},
		Role:          sql.NullInt64{Int64: int64(role), Valid: true},
		CalcIndex:     sql.NullInt64{Int64: int64(index), Valid: true},
		CalcFormula:   sql.NullString{String: string(jsonFormula), Valid: true},
		EffectiveTime: sql.NullTime{Time: effectiveTime, Valid: true},
		Operator:      sql.NullString{String: "system", Valid: true},
		CalcId:        sql.NullInt64{Int64: lastInsertId, Valid: true},
	})
	if err != nil {
		logx.Errorw("Insert failed", logx.Field("error", err))
		return err
	}
	return nil
}
