package calcrpclogic

import (
	"context"
	"database/sql"
	"encoding/json"
	"errors"
	"fmt"
	"slices"
	"time"

	"oa-server/app/oacenter/model/efficiency_calc"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type ModifyCalcFormulaLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewModifyCalcFormulaLogic(ctx context.Context, svcCtx *svc.ServiceContext) *ModifyCalcFormulaLogic {
	return &ModifyCalcFormulaLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *ModifyCalcFormulaLogic) ModifyCalcFormula(in *oa.ModifyCalcFormulaReq) (*oa.EmptyResponse, error) {

	_, listResp, err := l.svcCtx.EfficiencyCalcModel.List(l.ctx, &efficiency_calc.ListReq{
		Page:     1,
		PageSize: 999999,
		ID:       in.Id,
	})
	if err != nil {
		logx.Errorw("List failed", logx.Field("error", err))
		return nil, err
	}
	if len(listResp) == 0 {
		return nil, errors.New("计算公式不存在")
	}
	departmentMap, err := l.svcCtx.UserCache.GetDepartmentMap(l.ctx)
	if err != nil {
		logx.Errorw("GetDepartmentMap failed", logx.Field("error", err))
		return nil, err
	}
	if !slices.Contains(l.svcCtx.Config.EfficiencyCalc.Admin, in.Operator) {
		for _, v := range listResp {
			if departmentMap[v.DepartmentKey.String].Owner != in.Operator {
				return nil, fmt.Errorf("您无权限修改%s部门", departmentMap[v.DepartmentKey.String].DepartmentName)
			}
		}
	}
	effectiveTime, err := time.Parse("2006-01", in.EffectiveTime)
	if err != nil {
		logx.Errorw("Parse effective time failed", logx.Field("error", err))
		return nil, err
	}
	for _, v := range listResp {
		calcFormula, err := json.Marshal(in.CalcFormula)
		if err != nil {
			logx.Errorw("Marshal calc formula failed", logx.Field("error", err))
			return nil, err
		}
		err = l.svcCtx.EfficiencyCalcModel.Update(l.ctx, &efficiency_calc.XEfficiencyCalc{
			CalcFormula:         sql.NullString{String: string(calcFormula), Valid: true},
			EffectiveTime:       sql.NullTime{Time: effectiveTime, Valid: true},
			UpdateBy:            sql.NullString{String: in.Operator, Valid: true},
			Id:                  v.Id,
			DepartmentKey:       v.DepartmentKey,
			ParentDepartmentKey: v.ParentDepartmentKey,
			Role:                v.Role,
			CalcIndex:           v.CalcIndex,
		})
		if err != nil {
			logx.Errorw("Update failed", logx.Field("error", err))
			return nil, err
		}
		//  创建history
		_, err = l.svcCtx.EfficiencyCalcHistoryModel.Insert(l.ctx, &efficiency_calc.XEfficiencyCalcHistory{
			Operator:      sql.NullString{String: in.Operator, Valid: true},
			EffectiveTime: sql.NullTime{Time: effectiveTime, Valid: true},
			CalcFormula:   sql.NullString{String: string(calcFormula), Valid: true},
			CalcIndex:     sql.NullInt64{Int64: int64(v.CalcIndex.Int64), Valid: true},
			Role:          sql.NullInt64{Int64: int64(v.Role.Int64), Valid: true},
			CalcId:        sql.NullInt64{Int64: v.Id, Valid: true},
			DepartmentKey: sql.NullString{String: v.DepartmentKey.String, Valid: true},
		})
		if err != nil {
			logx.Errorw("Insert history failed", logx.Field("error", err))
			continue
		}
	}
	return &oa.EmptyResponse{}, nil
}
