package issuerpclogic

import (
	"context"
	"database/sql"
	"encoding/json"
	"errors"
	"time"

	"oa-server/app/oacenter/model"
	"oa-server/app/oacenter/model/efficiency_calc"
	"oa-server/app/oacenter/model/issue"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common"
	"oa-server/common/constant"
	"oa-server/common/format"
	"oa-server/common/period"

	"github.com/spf13/cast"
	"github.com/zeromicro/go-zero/core/logx"
)

type CreateIssueMetricsLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewCreateIssueMetricsLogic(ctx context.Context, svcCtx *svc.ServiceContext) *CreateIssueMetricsLogic {
	return &CreateIssueMetricsLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 创建issue指标
func (l *CreateIssueMetricsLogic) CreateIssueMetrics(in *oa.CreateIssueMetricsReq) (*oa.EmptyResponse, error) {
	userInfo, err := l.svcCtx.UserCache.GetUserByEmail(l.ctx, in.Employee)
	if err != nil {
		logx.Errorf("获取用户信息失败: %v", err)
		return nil, err
	}

	// 删除已存在数据
	err = l.svcCtx.IssueModel.DeleteByFiled(l.ctx, issue.DeleteReq{
		PeriodType: int(in.PeriodType),
		Employee:   in.Employee,
		Department: userInfo.SecondDepartmentList[0].DepartmentKey,
		DateFromTo: in.DateFromTo,
		Role:       in.Role,
	})
	if err != nil && !errors.Is(err, sql.ErrNoRows) {
		logx.Errorf("delete issue metrics error: %v", err)
		return nil, err
	}

	prodBugCount := 0
	nonProdBugCount := 0
	for _, info := range in.Infos {
		if info.Type == "fe" {
			prodBugCount = prodBugCount + int(info.Count)
		} else {
			nonProdBugCount = nonProdBugCount + int(info.Count)
		}
	}

	var lastMetrics *issue.IssueMetricsDTO
	periodList, err := period.GetPeriodList(int(in.PeriodType), in.DateFromTo, 2, period.PeriodDirectionForward)
	if err != nil {
		logx.Errorf("获取周期列表失败: %v", err)
	} else if len(periodList) > 1 {
		_, metricsList, err := l.svcCtx.IssueModel.List(l.ctx, &issue.ListReq{
			Page:       1,
			PageSize:   1,
			DateFromTo: []string{periodList[1]},
			PeriodType: int(in.PeriodType),
			Employee:   in.Employee,
			Role:       in.Role,
		})
		if err != nil {
			logx.Errorf("获取上期数据失败: %v", err)
		} else if len(metricsList) > 0 {
			lastMetrics = metricsList[0]
		}
	}
	currentBugEscapeMetrics, currentProdBugMetrics, currentNonProdBugMetrics, totalScore := l.calculateScore(prodBugCount, nonProdBugCount, in.Role, userInfo.SecondDepartmentList[0].DepartmentKey)

	if lastMetrics != nil {
		currentBugEscapeMetrics.Last = lastMetrics.BugEscape.Current
		currentProdBugMetrics.Last = lastMetrics.ProdBug.Current
		currentNonProdBugMetrics.Last = lastMetrics.NonProd.Current
	}

	issueDao := issue.XIssueMetrics{
		PeriodType: sql.NullInt64{Int64: in.PeriodType, Valid: true},
		Employee:   sql.NullString{String: in.Employee, Valid: true},
		Department: sql.NullString{String: userInfo.SecondDepartmentList[0].DepartmentKey, Valid: true},
		DateFromTo: sql.NullString{String: in.DateFromTo, Valid: true},
		ProjectKey: sql.NullString{String: in.ProjectKey, Valid: true},
		Role:       sql.NullInt64{Int64: int64(in.Role), Valid: true},
		ProdBug:    sql.NullString{String: string(common.Must(json.Marshal(currentProdBugMetrics))), Valid: true},
		NonProd:    sql.NullString{String: string(common.Must(json.Marshal(currentNonProdBugMetrics))), Valid: true},
		BugEscape:  sql.NullString{String: string(common.Must(json.Marshal(currentBugEscapeMetrics))), Valid: true},
		Infos:      sql.NullString{String: string(common.Must(json.Marshal(in.Infos))), Valid: true},
		Score:      sql.NullFloat64{Float64: totalScore, Valid: true},
	}
	_, err = l.svcCtx.IssueModel.Insert(l.ctx, &issueDao)
	if err != nil {
		logx.Errorf("create issue metrics error: %v", err)
		return nil, err
	}
	return &oa.EmptyResponse{}, nil
}

func (l *CreateIssueMetricsLogic) calculateScore(prodBugCount, nonProdBugCount int, role oa.Role, departmentKey string) (currentBugEscapeMetrics, currentProdBugMetrics, currentNonProdBugMetrics model.MetricInfo, totalScore float64) {
	currentBugEscapeMetrics = model.MetricInfo{
		Current: cast.ToFloat64(format.FormatFloatWithPrecision(float64(prodBugCount/(prodBugCount+nonProdBugCount)*100), 2)),
		Score:   calculateBugEscapeRateScore(float64(prodBugCount / (prodBugCount + nonProdBugCount) * 100)),
	}
	currentProdBugMetrics = model.MetricInfo{
		Current: float64(prodBugCount),
		Score:   calculateProdBugCountScore(prodBugCount),
	}
	currentNonProdBugMetrics = model.MetricInfo{
		Current: float64(nonProdBugCount),
		Score:   calculateNonProdBugCountScore(nonProdBugCount),
	}

	_, list, err := l.svcCtx.EfficiencyCalcHistoryModel.ListHistory(l.ctx, &efficiency_calc.ListHistoryReq{
		DepartmentKey:         departmentKey,
		Role:                  role,
		CalcIndex:             oa.CalcIndex_DELIVERY_QUALITY,
		EffectiveTimeLessThan: time.Now(),
	})
	if err != nil {
		logx.Errorf("获取打分指标失败: %v", err)
		return
	}
	if len(list) == 0 {
		logx.Infow("未获取到当前时间生效的打分指标", logx.Field("departmentKey", departmentKey), logx.Field("role", role))
		totalScore = 0
		return
	}
	calcFormula := list[0].CalcFormula
	for _, item := range calcFormula {
		switch item.Key {
		case constant.BugEscapeRate:
			if role == oa.Role_TEST {
				totalScore += currentBugEscapeMetrics.Score * cast.ToFloat64(item.Value)
			}
		case constant.ProdBugCount:
			totalScore += currentProdBugMetrics.Score * cast.ToFloat64(item.Value)
		case constant.NonProdBugCount:
			totalScore += currentNonProdBugMetrics.Score * cast.ToFloat64(item.Value)
		}
	}
	return currentBugEscapeMetrics, currentProdBugMetrics, currentNonProdBugMetrics, cast.ToFloat64(format.FormatFloatWithPrecision(totalScore, 2))

}

func calculateBugEscapeRateScore(value float64) float64 {
	score := 0
	switch {
	case value == 0:
		score = 100
	case value > 0 && value <= 5:
		score = 80
	case value > 5 && value <= 10:
		score = 60
	case value > 10:
		score = 10
	}
	return float64(score)
}

func calculateProdBugCountScore(value int) float64 {
	score := 0
	switch {
	case value == 0:
		score = 100
	case value > 0 && value <= 2:
		score = 60
	case value > 2 && value <= 5:
		score = 30
	case value > 5:
		score = 10
	}
	return float64(score)
}

func calculateNonProdBugCountScore(value int) float64 {
	score := 0
	switch {
	case value == 0:
		score = 100
	case value > 0 && value <= 5:
		score = 80
	case value > 5 && value <= 10:
		score = 60
	case value > 10:
		score = 30
	}
	return float64(score)
}
