package issuerpclogic

import (
	"context"
	"fmt"

	"oa-server/app/jobcenter/model"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/format"

	"github.com/spf13/cast"
	"github.com/zeromicro/go-zero/core/logx"
)

type DepartmentChartDataLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewDepartmentChartDataLogic(ctx context.Context, svcCtx *svc.ServiceContext) *DepartmentChartDataLogic {
	return &DepartmentChartDataLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取部门图表数据
func (l *DepartmentChartDataLogic) DepartmentChartData(in *oa.MetricsListReq) (*oa.ChartDataResp, error) {
	issueListService := NewDepartmentIssueListLogic(l.ctx, l.svcCtx)
	issueListResp, err := issueListService.DepartmentIssueList(&oa.IssueListReq{
		PageNum:       int64(in.Page),
		PageSize:      int64(in.PageSize),
		PeriodType:    in.PeriodType,
		DepartmentKey: in.DepartmentKey,
		DateFromTo:    in.DateFromTo,
		Employee:      in.Employee,
	})
	if err != nil {
		logx.Errorf("获取部门图表数据失败: %v", err)
		return nil, err
	}
	if len(issueListResp.List) == 0 {
		logx.Infof("没有找到符合条件的监控指标数据")
		return &oa.ChartDataResp{}, nil
	}
	for _, v := range issueListResp.List {
		v.ProdBugCount = cast.ToFloat64(format.FormatFloatWithPrecision(v.ProdBugCount, 2))
		v.NonProdBugCount = cast.ToFloat64(format.FormatFloatWithPrecision(v.NonProdBugCount, 2))
		v.BugEscapeRate = cast.ToFloat64(format.FormatFloatWithPrecision(v.BugEscapeRate, 2))
	}

	// 构建图表数据
	return l.buildChartData(issueListResp.List), nil
}

var indicators = []string{
	"线上BUG数量",
	"线下BUG数量",
	"BUG逃逸率",
}

// 构建图表数据
func (l *DepartmentChartDataLogic) buildChartData(metricsList []*oa.IssueList) *oa.ChartDataResp {
	userCache := model.NewUserCenterService(l.svcCtx.RedisClient)
	userInfo, err := userCache.GetUserByEmail(l.ctx, metricsList[0].Employee)
	if err != nil {
		logx.Errorf("获取用户信息失败: %v", err)
		return nil
	}
	// 指标顺序不要修改
	// 如果修改，后面的数据顺序也要修改

	// 雷达图
	radarData := &oa.RadarChartData{
		Indicator: indicators,
		Series:    []*oa.RadarSeries{{Data: []*oa.RadarSeriesData{}}},
	}

	// 折线、柱状图
	lineBarData := &oa.LineBarChartData{
		XAxis:  []string{},
		Series: []*oa.LineBarSeriesItem{},
	}
	//热力图
	heatmapData := &oa.HeatmapChartData{
		XAxis:  []string{},
		YAxis:  indicators[:2],
		Series: []*oa.HeatmapSeriesItem{},
	}
	if userInfo.UserRole == model.RoleTester {
		heatmapData.YAxis = append(heatmapData.YAxis, indicators[2])
	}

	for i, metrics := range metricsList {
		nickname := metrics.Nickname
		if nickname == "" {
			nickname = fmt.Sprintf("员工%d", i+1)
		}

		// 雷达图 - 所有数据添加到第一个Series中
		radarData.Series[0].Data = append(radarData.Series[0].Data, &oa.RadarSeriesData{
			Name: nickname,
			Value: []float64{
				metrics.ProdBugCount,
				metrics.NonProdBugCount,
				metrics.BugEscapeRate,
			},
		})

		// 折线、柱状图
		lineBarData.XAxis = append(lineBarData.XAxis, nickname)

		// 初始化所有系列
		if i == 0 {
			lineBarData.Series = append(lineBarData.Series,
				&oa.LineBarSeriesItem{Name: indicators[0], Data: []float64{}},
				&oa.LineBarSeriesItem{Name: indicators[1], Data: []float64{}},
			)
			if userInfo.UserRole == model.RoleTester {
				lineBarData.Series = append(lineBarData.Series,
					&oa.LineBarSeriesItem{Name: indicators[2], Data: []float64{}, YAxisIndex: 1},
				)
			}
		}
		lineBarData.Series[0].Data = append(lineBarData.Series[0].Data, float64(metrics.ProdBugCount))
		lineBarData.Series[1].Data = append(lineBarData.Series[1].Data, float64(metrics.NonProdBugCount))
		if userInfo.UserRole == model.RoleTester {
			lineBarData.Series[2].Data = append(lineBarData.Series[2].Data, float64(metrics.BugEscapeRate))
		}
		// 热力图
		heatmapData.XAxis = append(heatmapData.XAxis, nickname)
		heatValueList := []float64{
			float64(metrics.ProdBugCount),
			float64(metrics.NonProdBugCount),
		}
		if userInfo.UserRole == model.RoleTester {
			heatValueList = append(heatValueList, float64(metrics.BugEscapeRate))
		}
		for j, value := range heatValueList {
			heatmapData.Series = append(heatmapData.Series, &oa.HeatmapSeriesItem{
				X:     int32(j),
				Y:     int32(i),
				Value: value,
			})
		}
	}

	return &oa.ChartDataResp{
		RadarData:   radarData,
		LineBarData: lineBarData,
		HeatmapData: heatmapData,
	}
}
