package issuerpclogic

import (
	"context"

	"oa-server/app/jobcenter/model"
	"oa-server/app/oacenter/model/issue"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type DepartmentIssueListLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewDepartmentIssueListLogic(ctx context.Context, svcCtx *svc.ServiceContext) *DepartmentIssueListLogic {
	return &DepartmentIssueListLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取issue列表
func (l *DepartmentIssueListLogic) DepartmentIssueList(in *oa.IssueListReq) (*oa.IssueListResp, error) {
	total, list, err := l.svcCtx.IssueModel.List(l.ctx, &issue.ListReq{
		PeriodType: int(in.PeriodType),
		Department: in.DepartmentKey,
		DateFromTo: []string{in.DateFromTo},
		Employee:   *in.Employee,
		Page:       int(in.PageNum),
		PageSize:   int(in.PageSize),
	})
	if err != nil {
		logx.Errorw("IssueListLogic IssueList error", logx.Field("error", err), logx.Field("department", in.DepartmentKey))
		return nil, err
	}
  department, err := l.svcCtx.UserCache.GetDepartmentByKey(l.ctx, in.DepartmentKey)
  if err != nil {
    logx.Errorw("get department info error", logx.Field("error", err), logx.Field("department", in.DepartmentKey))
    return nil, err
  }
	issueList := make([]*oa.IssueList, 0)
	for _, v := range list {
		user, err := l.svcCtx.UserCache.GetUserByEmail(l.ctx, v.Employee.String)
		if err != nil {
			logx.Errorw("get user info error", logx.Field("error", err), logx.Field("department", in.DepartmentKey))
			continue
		}
		
		tmp := &oa.IssueList{
			Id:              v.Id,
			Nickname:        user.NickName,
			DepartmentName:  department.DepartmentName,
			ProdBugCount:    v.ProdBug.Current,
			NonProdBugCount: v.NonProd.Current,
			Score:           float32(v.Score.Float64),
			Employee:        v.Employee.String,
			Infos:           v.Infos,
		}
		if user.UserRole == model.RoleTester {
			tmp.BugEscapeRate = v.BugEscape.Current
		}
		issueList = append(issueList, tmp)
	}

	return &oa.IssueListResp{
		Total: total,
		List:  issueList,
	}, nil
}
