package issuerpclogic

import (
	"context"
	"errors"
	"fmt"

	"oa-server/app/jobcenter/model"
	"oa-server/app/oacenter/model/issue"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/format"
	"oa-server/common/period"

	"github.com/spf13/cast"
	"github.com/zeromicro/go-zero/core/logx"
)

type PersonalChartDataLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewPersonalChartDataLogic(ctx context.Context, svcCtx *svc.ServiceContext) *PersonalChartDataLogic {
	return &PersonalChartDataLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取个人图表数据
func (l *PersonalChartDataLogic) PersonalChartData(in *oa.PersonalMetricsReq) (*oa.ChartDataResp, error) {
	userInfo, err := l.svcCtx.UserCache.GetUserByEmail(l.ctx, *in.Employee)
	if err != nil {
		logx.Errorf("获取用户信息失败: %v", err)
		return nil, err
	}

	_, metricsList, err := l.svcCtx.IssueModel.List(l.ctx, &issue.ListReq{
		Page:       1,
		PageSize:   1,
		DateFromTo: []string{in.DateFromTo},
		PeriodType: int(in.PeriodType),
		Employee:   *in.Employee,
		Role:       oa.Role(userInfo.UserRole),
	})
	if err != nil {
		logx.Errorf("获取个人指标数据失败: %v", err)
		return nil, err
	}
	if len(metricsList) == 0 {
		logx.Infof("未找到个人指标数据")
		return nil, errors.New("未找到个人指标数据")
	}
	personalMetrics := metricsList[0]

	// 获取部门平均值
	personalMetrics.ProdBug.Current = cast.ToFloat64(format.FormatFloatWithPrecision(float64(personalMetrics.ProdBug.Current), 2))
	personalMetrics.NonProd.Current = cast.ToFloat64(format.FormatFloatWithPrecision(float64(personalMetrics.NonProd.Current), 2))
	personalMetrics.BugEscape.Current = cast.ToFloat64(format.FormatFloatWithPrecision(float64(personalMetrics.BugEscape.Current), 2))

	// 生成图数据
	switch *in.ChartType {
	case oa.ChartType_CHART_TYPE_LINE:
		return &oa.ChartDataResp{LineBarData: l.buildLineData(personalMetrics, userInfo.UserRole)}, nil
	case oa.ChartType_CHART_TYPE_RADAR:
		return &oa.ChartDataResp{RadarData: l.buildRadarData(personalMetrics, userInfo.UserRole)}, nil
	default:
		return &oa.ChartDataResp{}, errors.New("未知图表类型")
	}

}

type MetricsAverage struct {
	ProdBugCount    float64 // 生产bug数量
	NonProdBugCount float64 // 非生产bug数量
	BugEscapeRate   float64 // bug逃逸率
	// Score           float64 // 综合得分
}

// CalculateMetricsAverage 计算指标平均值
func CalculateMetricsAverage(metricsList []*issue.IssueMetricsDTO) (*MetricsAverage, error) {
	if len(metricsList) == 0 {
		return nil, fmt.Errorf("指标列表为空")
	}

	avg := &MetricsAverage{}
	count := float64(len(metricsList))

	for _, v := range metricsList {

		avg.ProdBugCount += v.ProdBug.Current
		avg.NonProdBugCount += v.NonProd.Current
		avg.BugEscapeRate += v.BugEscape.Current
		// avg.Score += dto.Score.Float64
	}

	// 计算平均值
	avg.ProdBugCount = cast.ToFloat64(format.FormatFloatWithPrecision(avg.ProdBugCount/count, 2))
	avg.NonProdBugCount = cast.ToFloat64(format.FormatFloatWithPrecision(avg.NonProdBugCount/count, 2))
	avg.BugEscapeRate = cast.ToFloat64(format.FormatFloatWithPrecision(avg.BugEscapeRate/count, 2))
	// avg.Score /= count

	return avg, nil
}

func (l *PersonalChartDataLogic) buildRadarData(personalMetrics *issue.IssueMetricsDTO, role model.UserRole) *oa.RadarChartData {
	// TODO 检查部门信息
	_, departmentMetricsList, err := l.svcCtx.IssueModel.List(l.ctx, &issue.ListReq{
		Page:       1,
		PageSize:   999999,
		DateFromTo: []string{personalMetrics.DateFromTo.String},
		PeriodType: int(personalMetrics.PeriodType.Int64),
		Department: personalMetrics.Department.String,
	})
	if err != nil {
		logx.Errorf("获取部门指标数据失败: %v", err)
		return nil
	}
	departmentMetrics, err := CalculateMetricsAverage(departmentMetricsList)
	if err != nil {
		logx.Errorf("计算部门指标平均值失败: %v", err)
		return nil
	}

	radarData := &oa.RadarChartData{}
	radarData.Indicator = indicators
	radarData.Series = []*oa.RadarSeries{
		{
			Data: []*oa.RadarSeriesData{
				{
					Name: "个人",
					Value: []float64{
						personalMetrics.ProdBug.Current,
						personalMetrics.NonProd.Current,
						personalMetrics.BugEscape.Current,
						// personalMetrics.Score.Float64,
					},
				},
				{
					Name: "部门",
					Value: []float64{
						departmentMetrics.ProdBugCount,
						departmentMetrics.NonProdBugCount,
						departmentMetrics.BugEscapeRate,
						// departmentMetrics.Score,
					},
				},
			},
		},
	}

	return radarData
}

func (l *PersonalChartDataLogic) buildLineData(personalMetrics *issue.IssueMetricsDTO, role model.UserRole) *oa.LineBarChartData {
	// 获取五个周期的值
	periodList, err := period.GetPeriodList(int(personalMetrics.PeriodType.Int64), personalMetrics.DateFromTo.String, 5, period.PeriodDirectionForward)
	if err != nil {
		logx.Errorf("获取周期列表失败: %v", err)
		return nil
	}

	// 反转周期列表
	for i, j := 0, len(periodList)-1; i < j; i, j = i+1, j-1 {
		periodList[i], periodList[j] = periodList[j], periodList[i]
	}

	_, metricsList, err := l.svcCtx.IssueModel.List(l.ctx, &issue.ListReq{
		Page:       1,
		PageSize:   10,
		DateFromTo: periodList,
		Department: personalMetrics.Department.String,
		Employee:   personalMetrics.Employee.String,
		PeriodType: int(personalMetrics.PeriodType.Int64),
		Role:       oa.Role(role),
	})
	if err != nil {
		logx.Errorf("获取周期列表失败: %v", err)
		return nil
	}

	metricsMap := make(map[string]*issue.IssueMetricsDTO)
	for _, metrics := range metricsList {
		metricsMap[metrics.DateFromTo.String] = metrics
	}

	lineData := &oa.LineBarChartData{
		XAxis: periodList,
	}

	lineData.Series = []*oa.LineBarSeriesItem{
		{Name: indicators[0], Data: make([]float64, len(periodList)), Type: "line"},
		{Name: indicators[1], Data: make([]float64, len(periodList)), Type: "line"},
	}
	if role == model.RoleTester {
		lineData.Series = append(lineData.Series, &oa.LineBarSeriesItem{Name: indicators[2], Data: make([]float64, len(periodList)), Type: "line"})
	}

	// 按周期顺序填充数据
	for i, period := range periodList {
		if dto, exists := metricsMap[period]; exists {
			lineData.Series[0].Data[i] = dto.ProdBug.Current
			lineData.Series[1].Data[i] = dto.NonProd.Current
			if role == model.RoleTester {
				lineData.Series[2].Data[i] = dto.BugEscape.Current
			}
		}
		// 如果该周期没有数据，保持默认值0
	}

	return lineData
}
