package issuerpclogic

import (
	"context"
	"fmt"

	"oa-server/app/jobcenter/model"
	"oa-server/app/oacenter/model/issue"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type PersonalIssueMetricsLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewPersonalIssueMetricsLogic(ctx context.Context, svcCtx *svc.ServiceContext) *PersonalIssueMetricsLogic {
	return &PersonalIssueMetricsLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取个人issue指标
func (l *PersonalIssueMetricsLogic) PersonalIssueMetrics(in *oa.IssuePersonalMetricsReq) (*oa.IssuePersonalMetricsResp, error) {
	if in.Employee == nil {
		return nil, fmt.Errorf("员工为空")
	}
	userInfo, err := l.svcCtx.UserCache.GetUserByEmail(l.ctx, *in.Employee)
	if err != nil {
		logx.Errorf("获取用户信息失败: %v", err)
		return nil, err
	}

	//  因为角色的原因，数据库里一个用户可能有多条数据，这里要区分角色查询
	_, issueMetrics, err := l.svcCtx.IssueModel.List(l.ctx, &issue.ListReq{
		Page:       1,
		PageSize:   1,
		DateFromTo: []string{in.DateFromTo},
		PeriodType: int(in.PeriodType),
		Employee:   *in.Employee,
		Role:       oa.Role(userInfo.UserRole),
	})
	if err != nil {
		logx.Errorw("获取个人issue指标失败", logx.Field("error", err), logx.Field("employee", in.Employee))
		return nil, fmt.Errorf("获取个人issue指标失败")
	}

	if len(issueMetrics) == 0 {
		return nil, fmt.Errorf("没有找到个人issue指标")
	}
	issueDto := issueMetrics[0]

	// 获取部门内所有用户指标数据
	_, departmentMetricsList, err := l.svcCtx.IssueModel.List(l.ctx, &issue.ListReq{
		Page:       1,
		PageSize:   999999,
		DateFromTo: []string{in.DateFromTo},
		PeriodType: int(in.PeriodType),
		Department: issueMetrics[0].Department.String,
		Role:       oa.Role(userInfo.UserRole),
	})
	if err != nil {
		logx.Errorf("获取部门指标数据失败: %v", err)
		return nil, err
	}
	if len(departmentMetricsList) == 0 {
		return nil, fmt.Errorf("没有找到部门指标数据")
	}

	// 计算部门内排名
	// 100 1
	// 100 1
	// 99 2
	// 99 2
	// 98 3
	// 97 4
	var rank int64 = 0
	var currentRank int64 = 1
	var currentScore float64 = departmentMetricsList[0].Score.Float64

	for _, metric := range departmentMetricsList {
		if metric.Score.Float64 != currentScore {
			currentRank = currentRank + 1
			currentScore = metric.Score.Float64
		}
		if metric.Employee.String == *in.Employee {
			rank = currentRank
			break
		}
	}

	resp := &oa.IssuePersonalMetricsResp{
		Id: issueMetrics[0].Id,
		ProdBug: &oa.MetricItemData{
			Current: issueDto.ProdBug.Current,
			Last:    issueDto.ProdBug.Last,
			Score:   float32(issueDto.ProdBug.Score),
		},
		NonProdBug: &oa.MetricItemData{
			Current: issueDto.NonProd.Current,
			Last:    issueDto.NonProd.Last,
			Score:   float32(issueDto.NonProd.Score),
		},
		Score:    float32(issueDto.Score.Float64),
		NickName: issueDto.Employee.String,
		Rank:     rank,
	}
	if userInfo.UserRole == model.RoleTester {
		resp.BugEscapeRate = &oa.MetricItemData{
			Current: issueDto.BugEscape.Current,
			Last:    issueDto.BugEscape.Last,
			Score:   float32(issueDto.BugEscape.Score),
		}
	}
	return resp, nil
}
