package monitorrpclogic

import (
	"context"
	"database/sql"
	"encoding/json"
	"oa-server/common"
	"oa-server/common/constant"
	"oa-server/common/format"
	"time"

	"github.com/spf13/cast"

	"oa-server/app/oacenter/model"
	"oa-server/app/oacenter/model/efficiency_calc"
	"oa-server/app/oacenter/model/monitor_metrics"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/period"

	"github.com/zeromicro/go-zero/core/logx"
)

type CreateMonitorMetricsLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewCreateMonitorMetricsLogic(ctx context.Context, svcCtx *svc.ServiceContext) *CreateMonitorMetricsLogic {
	return &CreateMonitorMetricsLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// CreateMonitorMetrics 创建监控指标数据
func (l *CreateMonitorMetricsLogic) CreateMonitorMetrics(in *oa.CreateMonitorMetricsReq) (*oa.EmptyResponse, error) {
	in.BizEventAssignmentRate = cast.ToFloat32(format.FormatFloatWithPrecision(float64(in.BizEventAssignmentRate), 2))
	in.BizRuleHitRate = cast.ToFloat32(format.FormatFloatWithPrecision(float64(in.BizRuleHitRate), 2))
	in.FailureRecurrenceRate = cast.ToFloat32(format.FormatFloatWithPrecision(float64(in.FailureRecurrenceRate), 2))
	err := l.svcCtx.MonitorMetricsModel.DeleteMetrics(l.ctx, in.PeriodType, in.DateFromTo, in.Employee)
	if err != nil {
		logx.Errorf("删除监控指标失败: %v", err)
		return nil, err
	}
	failureRecurrenceRate := model.MetricInfo{
		Current: cast.ToFloat64(format.FormatFloatWithPrecision(float64(in.FailureRecurrenceRate), 2)),
		Score:   calculateFailureRecurrenceRateScore(float64(in.FailureRecurrenceRate)).Score,
	}
	averageResolutionTime := model.MetricInfo{
		Current: cast.ToFloat64(format.FormatFloatWithPrecision(float64(in.AverageResolutionTime), 2)),
		Score:   calculateAverageResolutionTimeScore(float64(in.AverageResolutionTime)).Score,
	}
	averageProcessingTime := model.MetricInfo{
		Current: cast.ToFloat64(format.FormatFloatWithPrecision(float64(in.AverageProcessingTime), 2)),
		Score:   calculateAverageProcessingTimeScore(float64(in.AverageProcessingTime)).Score,
	}
	effectiveHandledEvents := model.MetricInfo{
		Current: float64(in.EffectiveHandledEvents),
		Score:   calculateEffectiveHandledEventsScore(float64(in.EffectiveHandledEvents), in.PeriodType).Score,
	}
	bizEventAssignmentRate := model.MetricInfo{
		Current: cast.ToFloat64(format.FormatFloatWithPrecision(float64(in.BizEventAssignmentRate), 2)),
		Score:   calculateBizEventAssignmentRateScore(float64(in.BizEventAssignmentRate)).Score,
	}
	bizRuleHitRate := model.MetricInfo{
		Current: cast.ToFloat64(format.FormatFloatWithPrecision(float64(in.BizRuleHitRate), 2)),
		Score:   calculateBizRuleHitRateScore(float64(in.BizRuleHitRate)).Score,
	}
	// 计算总分
	totalScore := l.calculateScore(in.Employee, failureRecurrenceRate.Score, averageResolutionTime.Score, averageProcessingTime.Score, effectiveHandledEvents.Score, bizEventAssignmentRate.Score, bizRuleHitRate.Score)

	userInfo, err := l.svcCtx.UserCache.GetUserByEmail(context.Background(), in.Employee)
	if err != nil {
		logx.Errorf("获取用户信息失败: %v", err)
		return nil, err
	}

	// 获取上一个周期的数据
	var lastMetrics *monitor_metrics.XMonitorMetrics
	periodList, err := period.GetPeriodList(int(in.PeriodType), in.DateFromTo, 2, period.PeriodDirectionForward)
	if err != nil {
		logx.Errorf("获取周期列表失败: %v", err)
	} else if len(periodList) > 1 {
		_, metricsList, err := l.svcCtx.MonitorMetricsModel.List(l.ctx, &monitor_metrics.ListReq{
			Page:          1,
			PageSize:      1,
			DateFromTo:    []string{periodList[1]},
			PeriodType:    in.PeriodType,
			DepartmentKey: userInfo.SecondDepartmentList[0].DepartmentKey,
			Employee:      in.Employee,
		})
		if err != nil {
			logx.Errorf("获取上期数据失败: %v", err)
		} else if len(metricsList) > 0 {
			lastMetrics = metricsList[0]
		}
	}
	// 如果有上期数据，设置last值
	if lastMetrics != nil {
		lastDTO, err := lastMetrics.ToDTO()
		if err == nil {
			failureRecurrenceRate.Last = lastDTO.FailureRecurrenceRate.Current
			averageResolutionTime.Last = lastDTO.AverageResolutionTime.Current
			averageProcessingTime.Last = lastDTO.AverageProcessingTime.Current
			effectiveHandledEvents.Last = lastDTO.EffectiveHandledEvents.Current
			bizEventAssignmentRate.Last = lastDTO.BizEventAssignmentRate.Current
			bizRuleHitRate.Last = lastDTO.BizRuleHitRate.Current
		}
	}

	currentMetrics := &monitor_metrics.XMonitorMetrics{
		TotalEvents:            sql.NullInt64{Valid: true, Int64: in.TotalEvents}, // 默认值为0
		Employee:               sql.NullString{Valid: true, String: in.Employee},
		NickName:               sql.NullString{Valid: true, String: userInfo.NickName},
		PeriodType:             sql.NullInt64{Valid: true, Int64: in.PeriodType},
		DateFromTo:             sql.NullString{Valid: true, String: in.DateFromTo},
		FailureRecurrenceRate:  sql.NullString{Valid: true, String: string(common.Must(json.Marshal(failureRecurrenceRate)))},
		AverageProcessingTime:  sql.NullString{Valid: true, String: string(common.Must(json.Marshal(averageProcessingTime)))},
		AverageResolutionTime:  sql.NullString{Valid: true, String: string(common.Must(json.Marshal(averageResolutionTime)))},
		EffectiveHandledEvents: sql.NullString{Valid: true, String: string(common.Must(json.Marshal(effectiveHandledEvents)))},
		BizRuleHitRate:         sql.NullString{Valid: true, String: string(common.Must(json.Marshal(bizRuleHitRate)))},
		BizEventAssignmentRate: sql.NullString{Valid: true, String: string(common.Must(json.Marshal(bizEventAssignmentRate)))},
		DepartmentKey:          sql.NullString{Valid: true, String: userInfo.SecondDepartmentList[0].DepartmentKey},
		DepartmentName:         sql.NullString{Valid: true, String: userInfo.SecondDepartmentList[0].DepartmentName},
		Score:                  sql.NullFloat64{Valid: true, Float64: totalScore},
	}
	// 保存到数据库
	_, err = l.svcCtx.MonitorMetricsModel.Insert(l.ctx, currentMetrics)
	if err != nil {
		return nil, err
	}

	return &oa.EmptyResponse{}, nil
}

func (l *CreateMonitorMetricsLogic) calculateScore(employee string, failureRecurrenceRateScore, averageResponseTimeScore, averageProcessingTimeScore, effectiveHandledEventsScore, bizEventAssignmentRateScore, bizRuleHitRateScore float64) (totalScore float64) {

	userInfo, err := l.svcCtx.UserCache.GetUserByEmail(context.Background(), employee)
	if err != nil {
		logx.Errorf("获取用户信息失败: %v", err)
		return
	}
	_, list, err := l.svcCtx.EfficiencyCalcHistoryModel.ListHistory(l.ctx, &efficiency_calc.ListHistoryReq{
		DepartmentKey:         userInfo.SecondDepartmentList[0].DepartmentKey,
		Role:                  oa.Role(userInfo.UserRole),
		CalcIndex:             oa.CalcIndex_MONITOR_SCORE,
		EffectiveTimeLessThan: time.Now(),
	})
	if err != nil {
		logx.Errorf("获取打分指标历史失败: %v", err)
		return
	}
	if len(list) == 0 {
		logx.Info("没有获取到当前日期生效的打分公式")
		totalScore = 0
		return
	}
	// 使用最新的公式
	calcFormula := list[0].CalcFormula

	leaderDpt, err := l.svcCtx.UserCache.GetLeaderDepartmentsMap(context.Background())
	if err != nil {
		logx.Errorf("获取部门leader信息失败: %v", err)
		return
	}
	for _, item := range calcFormula {
		switch item.Key {
		case constant.FailureRecurrenceRate:
			totalScore += failureRecurrenceRateScore * cast.ToFloat64(item.Value)
		case constant.AverageResponseTime:
			totalScore += averageResponseTimeScore * cast.ToFloat64(item.Value)
		case constant.AverageProcessingTime:
			totalScore += averageProcessingTimeScore * cast.ToFloat64(item.Value)
		case constant.EffectiveHandledEvents:
			totalScore += effectiveHandledEventsScore * cast.ToFloat64(item.Value)
		case constant.BizEventAssignmentRate:
			if _, ok := leaderDpt[employee]; ok {
				totalScore += bizEventAssignmentRateScore * cast.ToFloat64(item.Value)
			}
		case constant.BizRuleHitRate:
			if _, ok := leaderDpt[employee]; ok {
				totalScore += bizRuleHitRateScore * cast.ToFloat64(item.Value)
			}
		}
	}

	return cast.ToFloat64(format.FormatFloatWithPrecision(totalScore, 2))
}

// func getDepartmentKey(bizID uint64) string {
// 	switch bizID {
// 	//合约后端
// 	case 2:
// 		return "od-42587cc93efc7424fc612cbb5c380440"
// 	//	现货后端
// 	case 4:
// 		return "od-7d4c7cd8bc999a10c3c843f7832d60a4"
// 	//	 统一账号后端
// 	case 6:
// 		return "od-4b3571b6ad34e4f52baf1bddd7be7934"
// 	//	管理平台组-后端
// 	case 8:
// 		return "od-4b3571b6ad34e4f52baf1bddd7be7934"
// 	//钱包资产组
// 	case 10:
// 		return "od-d13bfe965375041d4f243979c617eb76"
// 	//	开放平台组-后端
// 	case 12:
// 		return "od-4b3571b6ad34e4f52baf1bddd7be7934"
// 	//	金融产品组-后端
// 	case 16:
// 		return "od-7d4c7cd8bc999a10c3c843f7832d60a4"
// 		//	运维组
// 	case 17:
// 		return "od-633c031bb4dcdbaae8191c33c0beb021"
// 	//	营销清算组-后端
// 	case 31:
// 		return "od-f8654ea176392abfe2dc3370afd2c3df"
// 	//	EP后端
// 	case 32:
// 		return "od-14f406f360dbef84e474a9075cafcd89"
// 	//中台架构组
// 	case 35:
// 		return "od-796a759e04a851c2b5c922872d3c7556"
// 	//	运营活动组-后端
// 	case 36:
// 		return "od-f8654ea176392abfe2dc3370afd2c3df"
// 	default:
// 		return "未知部门"
// 	}
// }

// 计算故障复发率分数
func calculateFailureRecurrenceRateScore(value float64) model.MetricInfo {
	score := 0.0
	switch {
	case value > 0 && value <= 5:
		score = 60
	case value > 5 && value <= 10:
		score = 30
	}
	return model.MetricInfo{
		Current: value,
		Score:   score,
	}
}

// 计算平均解决时间分数
func calculateAverageResolutionTimeScore(value float64) model.MetricInfo {
	score := 0.0
	switch {
	case value <= 14400 && value > 0: // 4h
		score = 100
	case value > 14400 && value <= 43200: // 12h
		score = 80
	case value > 43200 && value <= 86400: // 24h
		score = 60
	case value > 86400 && value <= 259200: // 72h
		score = 40
	case value > 259200 && value <= 432000: // 120h
		score = 20
	case value > 432000: // >120h
		score = 10
	}
	return model.MetricInfo{
		Current: value,
		Score:   score,
	}
}

// 计算平均响应时间分数
func calculateAverageProcessingTimeScore(value float64) model.MetricInfo {
	score := 0.0
	switch {
	case value > 0 && value <= 1200: // 20分钟
		score = 100
	case value > 1200 && value <= 3600: // 60分钟
		score = 80
	case value > 3600 && value <= 7200: // 120分钟
		score = 60
	case value > 7200 && value <= 14400: // 240分钟
		score = 30
	case value > 14400 && value <= 36000: // 600分钟
		score = 10
	}
	return model.MetricInfo{
		Current: value,
		Score:   score,
	}
}

// 计算事件分配率分数
func calculateBizEventAssignmentRateScore(value float64) model.MetricInfo {
	score := 0.0
	switch {
	case value >= 60:
		score = 100
	case value >= 30:
		score = 70
	case value >= 10:
		score = 30
	}
	return model.MetricInfo{
		Current: value,
		Score:   score,
	}
}

// 计算有效处理事件数分数
func calculateEffectiveHandledEventsScore(value float64, periodType int64) model.MetricInfo {
	// 根据周期类型调整阈值
	threshold := 10.0
	switch periodType {
	case 2: // 月
		threshold = 40
	case 3: // 双月
		threshold = 80
	case 4: // 季度
		threshold = 120
	}

	score := 0.0
	switch {
	case value >= threshold:
		score = 100
	case value >= threshold*0.8:
		score = 90
	case value >= threshold*0.6:
		score = 80
	case value >= threshold*0.4:
		score = 70
	case value >= threshold*0.2:
		score = 50
	case value >= threshold*0.1:
		score = 30
	}
	return model.MetricInfo{
		Current: value,
		Score:   score,
	}
}

// 计算规则命中率分数
func calculateBizRuleHitRateScore(value float64) model.MetricInfo {
	score := 0.0
	switch {
	case value >= 10:
		score = 100
	case value >= 8:
		score = 90
	case value >= 6:
		score = 80
	case value >= 4:
		score = 70
	case value >= 2:
		score = 50
	case value >= 1:
		score = 30
	case value >= 0:
		score = 10
	}
	return model.MetricInfo{
		Current: value,
		Score:   score,
	}
}
