package monitorrpclogic

import (
	"context"
	"fmt"

	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/format"

	"github.com/spf13/cast"
	"github.com/zeromicro/go-zero/core/logx"
)

type GetMonitorChartDataLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewGetMonitorChartDataLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetMonitorChartDataLogic {
	return &GetMonitorChartDataLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取监控图表数据
func (l *GetMonitorChartDataLogic) GetMonitorChartData(in *oa.MetricsListReq) (*oa.ChartDataResp, error) {

	metricsLogic := NewGetMonitorMetricsListLogic(l.ctx, l.svcCtx)
	metricsResp, err := metricsLogic.GetMonitorMetricsList(in)
	if err != nil {
		logx.Errorf("获取监控指标列表失败: %v", err)
		return nil, fmt.Errorf("获取监控图表数据失败: %v", err)
	}

	// 检查是否有有效数据
	if len(metricsResp.List) == 0 {
		logx.Infof("没有找到符合条件的监控指标数据")
		return &oa.ChartDataResp{}, nil
	}
	for _, v := range metricsResp.List {
		v.BizRuleHitRate = cast.ToFloat64(format.FormatFloatWithPrecision(v.BizRuleHitRate, 2))
		v.BizEventAssignmentRate = cast.ToFloat64(format.FormatFloatWithPrecision(v.BizEventAssignmentRate, 2))
		v.FailureRecurrenceRate = cast.ToFloat64(format.FormatFloatWithPrecision(v.FailureRecurrenceRate, 2))
	}

	// 构建图表数据
	return l.buildChartData(metricsResp.List), nil
}

var indicators = []string{
	"有效处理事件",
	"平均处理时间(s)",
	"平均响应时间(s)",
	"故障复发率(%)",
	"事件分配率(%)",
	"规则命中率(%)",
}

// 构建图表数据
func (l *GetMonitorChartDataLogic) buildChartData(metricsList []*oa.MetricsInfo) *oa.ChartDataResp {
	// 指标顺序不要修改
	// 如果修改，后面的数据顺序也要修改

	// 雷达图
	radarData := &oa.RadarChartData{
		Indicator: indicators,
		Series:    []*oa.RadarSeries{{Data: []*oa.RadarSeriesData{}}},
	}
	// 折线、柱状图
	lineBarData := &oa.LineBarChartData{
		XAxis:  []string{},
		Series: []*oa.LineBarSeriesItem{},
	}
	//热力图
	heatmapData := &oa.HeatmapChartData{
		XAxis:  []string{},
		YAxis:  indicators,
		Series: []*oa.HeatmapSeriesItem{},
	}
	for i, metrics := range metricsList {
		nickname := metrics.Nickname
		if nickname == "" {
			nickname = fmt.Sprintf("员工%d", i+1)
		}

		// 雷达图 - 所有数据添加到第一个Series中
		radarData.Series[0].Data = append(radarData.Series[0].Data, &oa.RadarSeriesData{
			Name: nickname,
			Value: []float64{
				float64(metrics.EffectiveHandledEvents),
				float64(metrics.AverageProcessingTime),
				float64(metrics.AverageResolutionTime),
				float64(metrics.FailureRecurrenceRate),
				float64(metrics.BizEventAssignmentRate),
				float64(metrics.BizRuleHitRate),
			},
		})

		// 折线、柱状图
		lineBarData.XAxis = append(lineBarData.XAxis, nickname)

		// 初始化所有系列
		if i == 0 {
			lineBarData.Series = append(lineBarData.Series,
				&oa.LineBarSeriesItem{Name: indicators[0], Data: []float64{}, YAxisIndex: 1},
				&oa.LineBarSeriesItem{Name: indicators[1], Data: []float64{}},
				&oa.LineBarSeriesItem{Name: indicators[2], Data: []float64{}},
				&oa.LineBarSeriesItem{Name: indicators[3], Data: []float64{}, YAxisIndex: 1},
				&oa.LineBarSeriesItem{Name: indicators[4], Data: []float64{}, YAxisIndex: 1},
				&oa.LineBarSeriesItem{Name: indicators[5], Data: []float64{}, YAxisIndex: 1},
			)
		}
		lineBarData.Series[0].Data = append(lineBarData.Series[0].Data, float64(metrics.EffectiveHandledEvents))
		lineBarData.Series[1].Data = append(lineBarData.Series[1].Data, float64(metrics.AverageResolutionTime))
		lineBarData.Series[2].Data = append(lineBarData.Series[2].Data, float64(metrics.AverageProcessingTime))
		lineBarData.Series[3].Data = append(lineBarData.Series[3].Data, float64(metrics.FailureRecurrenceRate))
		lineBarData.Series[4].Data = append(lineBarData.Series[4].Data, float64(metrics.BizEventAssignmentRate))
		lineBarData.Series[5].Data = append(lineBarData.Series[5].Data, float64(metrics.BizRuleHitRate))

		// 热力图
		heatmapData.XAxis = append(heatmapData.XAxis, nickname)
		for j, value := range []float64{
			float64(metrics.EffectiveHandledEvents),
			float64(metrics.AverageProcessingTime),
			float64(metrics.AverageResolutionTime),
			float64(metrics.FailureRecurrenceRate),
			float64(metrics.BizEventAssignmentRate),
			float64(metrics.BizRuleHitRate),
		} {
			heatmapData.Series = append(heatmapData.Series, &oa.HeatmapSeriesItem{
				X:     int32(j),
				Y:     int32(i),
				Value: value,
			})
		}
	}

	return &oa.ChartDataResp{
		RadarData:   radarData,
		LineBarData: lineBarData,
		HeatmapData: heatmapData,
	}
}
