package monitorrpclogic

import (
	"context"
	"fmt"
	"oa-server/common/format"
	"oa-server/common/period"

	"github.com/pkg/errors"
	"github.com/spf13/cast"


	"oa-server/app/oacenter/model/monitor_metrics"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type GetPersonalChartDataLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewGetPersonalChartDataLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetPersonalChartDataLogic {
	return &GetPersonalChartDataLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// GetPersonalChartData 获取个人图表数据
func (l *GetPersonalChartDataLogic) GetPersonalChartData(in *oa.PersonalMetricsReq) (*oa.ChartDataResp, error) {
	_, metricsList, err := l.svcCtx.MonitorMetricsModel.List(l.ctx, &monitor_metrics.ListReq{
		Page:       1,
		PageSize:   1,
		DateFromTo: []string{in.DateFromTo},
		PeriodType: int64(in.PeriodType),
		Employee:   *in.Employee,
	})
	if err != nil {
		logx.Errorf("获取个人指标数据失败: %v", err)
		return nil, err
	}
	if len(metricsList) == 0 {
		logx.Infof("未找到个人指标数据")
		return nil, errors.New("未找到个人指标数据")
	}
	personalMetrics, err := metricsList[0].ToDTO()
	if err != nil {
		logx.Errorf("转换个人指标数据失败: %v", err)
		return nil, err
	}

	// 获取部门平均值
	personalMetrics.BizRuleHitRate.Current = cast.ToFloat64(format.FormatFloatWithPrecision(float64(personalMetrics.BizRuleHitRate.Current), 2))
	personalMetrics.BizEventAssignmentRate.Current = cast.ToFloat64(format.FormatFloatWithPrecision(float64(personalMetrics.BizEventAssignmentRate.Current), 2))
	personalMetrics.FailureRecurrenceRate.Current = cast.ToFloat64(format.FormatFloatWithPrecision(float64(personalMetrics.FailureRecurrenceRate.Current), 2))

	// 生成图数据
	switch *in.ChartType {
	case oa.ChartType_CHART_TYPE_LINE:
		return &oa.ChartDataResp{LineBarData: l.buildLineData(personalMetrics)}, nil
	case oa.ChartType_CHART_TYPE_RADAR:
		return &oa.ChartDataResp{RadarData: l.buildRadarData(personalMetrics)}, nil
	default:
		return &oa.ChartDataResp{}, errors.New("未知图表类型")
	}

}

type MetricsAverage struct {
	EffectiveHandledEvents float64 // 有效处理事件
	AverageProcessingTime  float64 // 平均处理时间
	AverageResolutionTime  float64 // 平均响应时间
	FailureRecurrenceRate  float64 // 故障复发率
	BizEventAssignmentRate float64 // 事件分配率
	BizRuleHitRate         float64 // 规则命中率
	Score                  float64 // 综合得分
}

// CalculateMetricsAverage 计算指标平均值
func CalculateMetricsAverage(metricsList []*monitor_metrics.XMonitorMetrics) (*MetricsAverage, error) {
	if len(metricsList) == 0 {
		return nil, fmt.Errorf("指标列表为空")
	}

	avg := &MetricsAverage{}
	count := float64(len(metricsList))

	for _, v := range metricsList {
		dto, err := v.ToDTO()
		if err != nil {
			return nil, fmt.Errorf("转换指标数据失败: %v", err)
		}

		avg.EffectiveHandledEvents += dto.EffectiveHandledEvents.Current
		avg.AverageProcessingTime += dto.AverageProcessingTime.Current
		avg.AverageResolutionTime += dto.AverageResolutionTime.Current
		avg.FailureRecurrenceRate += dto.FailureRecurrenceRate.Current
		avg.BizEventAssignmentRate += dto.BizEventAssignmentRate.Current
		avg.BizRuleHitRate += dto.BizRuleHitRate.Current
		avg.Score += dto.Score.Float64
	}

	// 计算平均值
	avg.EffectiveHandledEvents /= count
	avg.AverageProcessingTime /= count
	avg.AverageResolutionTime /= count
	avg.FailureRecurrenceRate /= count
	avg.BizEventAssignmentRate /= count
	avg.BizRuleHitRate /= count
	avg.Score /= count

	return avg, nil
}

func (l *GetPersonalChartDataLogic) buildRadarData(personalMetrics *monitor_metrics.MonitorMetricsDTO) *oa.RadarChartData {
	// TODO 检查部门信息
	_, departmentMetricsList, err := l.svcCtx.MonitorMetricsModel.List(l.ctx, &monitor_metrics.ListReq{
		Page:          1,
		PageSize:      999999,
		DateFromTo:    []string{personalMetrics.DateFromTo.String},
		PeriodType:    personalMetrics.PeriodType.Int64,
		DepartmentKey: personalMetrics.DepartmentKey.String,
	})
	if err != nil {
		logx.Errorf("获取部门指标数据失败: %v", err)
		return nil
	}
	departmentMetrics, err := CalculateMetricsAverage(departmentMetricsList)
	if err != nil {
		logx.Errorf("计算部门指标平均值失败: %v", err)
		return nil
	}

	radarData := &oa.RadarChartData{}
	radarData.Indicator = indicators
	radarData.Series = []*oa.RadarSeries{
		{
			Data: []*oa.RadarSeriesData{
				{
					Name: "个人",
					Value: []float64{
						personalMetrics.EffectiveHandledEvents.Current,
						personalMetrics.AverageResolutionTime.Current,
						personalMetrics.AverageProcessingTime.Current,
						personalMetrics.FailureRecurrenceRate.Current,
						personalMetrics.BizEventAssignmentRate.Current,
						personalMetrics.BizRuleHitRate.Current,
					},
				},
				{
					Name: "部门",
					Value: []float64{
						departmentMetrics.EffectiveHandledEvents,
						departmentMetrics.AverageResolutionTime,
						departmentMetrics.AverageProcessingTime,
						cast.ToFloat64(format.FormatFloatWithPrecision(departmentMetrics.FailureRecurrenceRate, 2)),
						cast.ToFloat64(format.FormatFloatWithPrecision(departmentMetrics.BizEventAssignmentRate, 2)),
						cast.ToFloat64(format.FormatFloatWithPrecision(departmentMetrics.BizRuleHitRate, 2)),
					},
				},
			},
		},
	}
	return radarData
}

func (l *GetPersonalChartDataLogic) buildLineData(personalMetrics *monitor_metrics.MonitorMetricsDTO) *oa.LineBarChartData {
	// 获取五个周期的值
	periodList, err := period.GetPeriodList(int(personalMetrics.PeriodType.Int64), personalMetrics.DateFromTo.String, 5, period.PeriodDirectionForward)
	if err != nil {
		logx.Errorf("获取周期列表失败: %v", err)
		return nil
	}

	// 反转周期列表
	for i, j := 0, len(periodList)-1; i < j; i, j = i+1, j-1 {
		periodList[i], periodList[j] = periodList[j], periodList[i]
	}

	_, metricsList, err := l.svcCtx.MonitorMetricsModel.List(l.ctx, &monitor_metrics.ListReq{
		Page:       1,
		PageSize:   10,
		DateFromTo: periodList,
		Employee:   personalMetrics.Employee.String,
		PeriodType: personalMetrics.PeriodType.Int64,
	})
	if err != nil {
		logx.Errorf("获取周期列表失败: %v", err)
		return nil
	}

	metricsMap := make(map[string]*monitor_metrics.MonitorMetricsDTO)
	for _, metrics := range metricsList {
		dto, err := metrics.ToDTO()
		if err != nil {
			logx.Errorf("转换指标数据失败: %v", err)
			continue
		}
		metricsMap[dto.DateFromTo.String] = dto
	}

	lineData := &oa.LineBarChartData{
		XAxis: periodList,
	}

	lineData.Series = []*oa.LineBarSeriesItem{
		{Name: indicators[0], Data: make([]float64, len(periodList)), YAxisIndex: 1, Type: "line"},
		{Name: indicators[1], Data: make([]float64, len(periodList)), Type: "line"},
		{Name: indicators[2], Data: make([]float64, len(periodList)), Type: "line"},
		{Name: indicators[3], Data: make([]float64, len(periodList)), YAxisIndex: 1, Type: "line"},
		{Name: indicators[4], Data: make([]float64, len(periodList)), YAxisIndex: 1, Type: "line"},
		{Name: indicators[5], Data: make([]float64, len(periodList)), YAxisIndex: 1, Type: "line"},
	}

	// 按周期顺序填充数据
	for i, period := range periodList {
		if dto, exists := metricsMap[period]; exists {
			lineData.Series[0].Data[i] = dto.EffectiveHandledEvents.Current
			lineData.Series[1].Data[i] = dto.AverageResolutionTime.Current
			lineData.Series[2].Data[i] = dto.AverageProcessingTime.Current
			lineData.Series[3].Data[i] = dto.FailureRecurrenceRate.Current
			lineData.Series[4].Data[i] = dto.BizEventAssignmentRate.Current
			lineData.Series[5].Data[i] = dto.BizRuleHitRate.Current
		}
		// 如果该周期没有数据，保持默认值0
	}

	return lineData
}
