package monitorrpclogic

import (
	"context"
	"fmt"
	"oa-server/app/oacenter/model/monitor_metrics"

	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type GetPersonalMetricsLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewGetPersonalMetricsLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetPersonalMetricsLogic {
	return &GetPersonalMetricsLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取个人指标数据
func (l *GetPersonalMetricsLogic) GetPersonalMetrics(in *oa.PersonalMetricsReq) (*oa.PersonalMetricsResp, error) {
	if in.Employee == nil {
		return nil, fmt.Errorf("员工为空")
	}

	// 获取个人指标数据
	_, metricsList, err := l.svcCtx.MonitorMetricsModel.List(l.ctx, &monitor_metrics.ListReq{
		Page:       1,
		PageSize:   1,
		DateFromTo: []string{in.DateFromTo},
		PeriodType: int64(in.PeriodType),
		Employee:   *in.Employee,
	})
	if err != nil {
		logx.Errorf("获取个人指标数据失败: %v", err)
		return nil, err
	}
	if len(metricsList) == 0 {
		logx.Infof("未找到个人指标数据")
		return &oa.PersonalMetricsResp{}, fmt.Errorf("未找到%s指标数据", *in.Employee)
	}

	// 获取部门内所有用户指标数据
	_, departmentMetricsList, err := l.svcCtx.MonitorMetricsModel.List(l.ctx, &monitor_metrics.ListReq{
		Page:          1,
		PageSize:      999999,
		DateFromTo:    []string{in.DateFromTo},
		PeriodType:    int64(in.PeriodType),
		DepartmentKey: metricsList[0].DepartmentKey.String,
		OrderBy:       "score DESC",
	})
	if err != nil {
		logx.Errorf("获取部门指标数据失败: %v", err)
		return nil, err
	}

	// 计算部门内排名
	// 100 1
	// 100 1
	// 99 2
	// 99 2
	// 98 3
	// 97 4
	var rank int64 = 0
	var currentRank int64 = 1
	var currentScore float64 = departmentMetricsList[0].Score.Float64

	for _, metric := range departmentMetricsList {
		if metric.Score.Float64 != currentScore {
			currentRank = currentRank + 1
			currentScore = metric.Score.Float64
		}
		if metric.Employee.String == *in.Employee {
			rank = currentRank
			break
		}
	}

	personalMetrics, err := metricsList[0].ToDTO()
	if err != nil {
		logx.Errorf("获取个人指标数据失败: %v", err)
		return nil, err
	}

	resp := &oa.PersonalMetricsResp{
		Id: personalMetrics.Id,
		FailureRecurrenceRate: &oa.MetricItemData{
			Current: personalMetrics.FailureRecurrenceRate.Current,
			Last:    personalMetrics.FailureRecurrenceRate.Last,
			Score:   float32(personalMetrics.FailureRecurrenceRate.Score),
		},
		AverageProcessingTime: &oa.MetricItemData{
			Current: personalMetrics.AverageProcessingTime.Current,
			Last:    personalMetrics.AverageProcessingTime.Last,
			Score:   float32(personalMetrics.AverageProcessingTime.Score),
		},
		AverageResolutionTime: &oa.MetricItemData{
			Current: personalMetrics.AverageResolutionTime.Current,
			Last:    personalMetrics.AverageResolutionTime.Last,
			Score:   float32(personalMetrics.AverageResolutionTime.Score),
		},
		EffectiveHandledEvents: &oa.MetricItemData{
			Current: personalMetrics.EffectiveHandledEvents.Current,
			Last:    personalMetrics.EffectiveHandledEvents.Last,
			Score:   float32(personalMetrics.EffectiveHandledEvents.Score),
		},
		BizEventAssignmentRate: &oa.MetricItemData{
			Current: personalMetrics.BizEventAssignmentRate.Current,
			Last:    personalMetrics.BizEventAssignmentRate.Last,
			Score:   float32(personalMetrics.BizEventAssignmentRate.Score),
		},
		BizRuleHitRate: &oa.MetricItemData{
			Current: personalMetrics.BizRuleHitRate.Current,
			Last:    personalMetrics.BizRuleHitRate.Last,
			Score:   float32(personalMetrics.BizRuleHitRate.Score),
		},
		Score:    float32(personalMetrics.Score.Float64),
		Nickname: personalMetrics.NickName.String,
		Rank:     rank,
	}
	return resp, nil
}
