package okrrpclogic

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"oa-server/app/oacenter/model/okr"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"time"

	"github.com/zeromicro/go-zero/core/logx"
)

type AlignAddLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewAlignAddLogic(ctx context.Context, svcCtx *svc.ServiceContext) *AlignAddLogic {
	return &AlignAddLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *AlignAddLogic) AlignAdd(in *oa.AlignAddReq) (*oa.EmptyResponse, error) {
	err := l.svcCtx.OkrTaskModel.TransCtx(l.ctx, func(ctx context.Context, session sqlx.Session) error {
		entities, err := l.svcCtx.OkrTaskModel.GetEntities(ctx, session, []int64{in.EntityId, in.AlignWithEntityId})
		if err != nil {
			return fmt.Errorf("OkrTaskModel.GetEntities: %w", err)
		}
		if len(entities) < 2 {
			return fmt.Errorf("OkrTaskModel.GetEntities: not found %v", []int64{in.EntityId, in.AlignWithEntityId})
		}
		period, err := okr.NewXOkrPeriodModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, entities[0].PeriodId)
		if err != nil {
			return fmt.Errorf("NewXOkrPeriodModel.FindOne: %w", err)
		}
		var entity, alignWithEntity *okr.XOKrTask
		for _, e := range entities {
			if e.EntityId == in.EntityId {
				entity = e
			}
			if e.EntityId == in.AlignWithEntityId {
				alignWithEntity = e
			}
		}
		if entity.Owner != in.UserEmail {
			return errors.New("你没有权限编辑")
		}
		if alignWithEntity.Owner == in.UserEmail {
			return errors.New("你不能对齐自己的条目")
		}

		alignment, _ := okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).FindOneByEntityIdAlignWithEntity(ctx, in.EntityId, in.AlignWithEntityId)
		if alignment != nil {
			if alignment.Intended == 1 {
				return errors.New("已经发起过对齐了")
			} else {
				return errors.New("已经对齐了")
			}
		}

		// 已经确认过的OKR不允许修改
		if period.ApprovalStatus == int64(oa.OkrStatus_OKR_STATUS_ACKED) {
			return fmt.Errorf("用户 %s 的okr已确认", period.Owner)
		}
		period2, err := okr.NewXOkrPeriodModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, alignWithEntity.PeriodId)
		if err != nil {
			return fmt.Errorf("NewXOkrPeriodModel.FindOne: %w", err)
		}
		// 已经确认过的OKR不允许修改
		if period2.ApprovalStatus == int64(oa.OkrStatus_OKR_STATUS_ACKED) {
			return fmt.Errorf("用户 %s 的okr已确认", period2.Owner)
		}

		entityAndTheirAncestors, err := l.svcCtx.OkrTaskModel.GetEntityRootPaths(ctx, session, []int64{in.EntityId})
		if err != nil {
			return fmt.Errorf("OkrTaskModel.GetEntityRootPaths: %w", err)
		}
		alignWithWhomAndTheirAncestors, err := l.svcCtx.OkrTaskModel.GetEntityRootPaths(ctx, session, []int64{in.AlignWithEntityId})
		if err != nil {
			return fmt.Errorf("OkrTaskModel.GetEntityRootPaths: %w", err)
		}
		content := okr.MsgContentAlignReq{
			MsgContentAlign: okr.MsgContentAlign{
				EntityId:                  in.EntityId,
				AlignWithEntityId:         in.AlignWithEntityId,
				ReqBy:                     entity.Owner,
				ReqHandleBy:               alignWithEntity.Owner,
				ReqHandleResult:           0,
				EntityPath:                entityAndTheirAncestors,
				AlignWithEntityEntityPath: alignWithWhomAndTheirAncestors,
			},
		}
		jsonContent, _ := json.Marshal(content)
		msg := &okr.XOkrMsg{
			Id:             0,
			FromUser:       entity.Owner,
			ToUser:         alignWithEntity.Owner,
			Type:           1,
			Content:        string(jsonContent),
			ApprovalStatus: 1,
			StartDate:      period.StartDate,
			EndDate:        period.EndDate,
			CreatedAt:      time.Time{},
			UpdatedAt:      time.Time{},
		}
		_, err = okr.NewXOkrMsgModel(sqlx.NewSqlConnFromSession(session)).Insert(ctx, msg)
		if err != nil {
			return fmt.Errorf("NewXOkrMsgModel.Insert: %w", err)
		}
		intendedAlignmentRecord := &okr.XOkrAlignment{
			Id:                0,
			EntityId:          in.EntityId,
			AlignWithEntity:   in.AlignWithEntityId,
			AlignWithUser:     alignWithEntity.Owner,
			AlignWithPosition: 0, //todo
			AlignWithWeight:   0, //todo
			Intended:          1,
			ByAssign:          0,
			CreatedAt:         time.Time{},
			UpdatedAt:         time.Time{},
		}
		_, err = okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).Insert(ctx, intendedAlignmentRecord)
		if err != nil {
			return fmt.Errorf("NewXOkrAlignmentModel.Insert: %w", err)
		}
		return nil
	})
	if err != nil {
		return nil, err
	}

	return &oa.EmptyResponse{}, nil
}
