package okrrpclogic

import (
	"context"
	"errors"
	"fmt"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"oa-server/app/oacenter/model/okr"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type AlignDelLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewAlignDelLogic(ctx context.Context, svcCtx *svc.ServiceContext) *AlignDelLogic {
	return &AlignDelLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *AlignDelLogic) AlignDel(in *oa.AlignDelReq) (*oa.EmptyResponse, error) {
	err := l.svcCtx.OkrTaskModel.TransCtx(l.ctx, func(ctx context.Context, session sqlx.Session) error {
		entities, err := l.svcCtx.OkrTaskModel.GetEntities(ctx, session, []int64{in.EntityId, in.AlignWithEntityId})
		if err != nil {
			return fmt.Errorf("OkrTaskModel.GetEntities: %w", err)
		}
		if len(entities) < 2 {
			return fmt.Errorf("OkrTaskModel.GetEntities: not found %v", []int64{in.EntityId, in.AlignWithEntityId})
		}
		var entity, alignWithEntity *okr.XOKrTask
		for _, e := range entities {
			if e.EntityId == in.EntityId {
				entity = e
			}
			if e.EntityId == in.AlignWithEntityId {
				alignWithEntity = e
			}
		}
		if entity.Owner != in.UserEmail && alignWithEntity.Owner != in.UserEmail {
			return errors.New("你没有权限编辑")
		}
		alignment, err := okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).FindOneByEntityIdAlignWithEntity(ctx, in.EntityId, in.AlignWithEntityId)
		if err != nil {
			if errors.Is(err, sqlx.ErrNotFound) {
				return nil
			}
			return fmt.Errorf("NewXOkrAlignmentModel.FindOneByEntityIdAlignWithEntity: %w", err)
		}

		period, err := okr.NewXOkrPeriodModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, entity.PeriodId)
		if err != nil {
			return fmt.Errorf("NewXOkrPeriodModel.FindOne: %w", err)
		}
		// 已经确认过的OKR不允许修改
		if period.ApprovalStatus == int64(oa.OkrStatus_OKR_STATUS_ACKED) {
			return fmt.Errorf("用户 %s 的okr已确认", period.Owner)
		}
		period2, err := okr.NewXOkrPeriodModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, alignWithEntity.PeriodId)
		if err != nil {
			return fmt.Errorf("NewXOkrPeriodModel.FindOne: %w", err)
		}
		// 已经确认过的OKR不允许修改
		if period2.ApprovalStatus == int64(oa.OkrStatus_OKR_STATUS_ACKED) {
			return fmt.Errorf("用户 %s 的okr已确认", period2.Owner)
		}

		err = okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).Delete(ctx, alignment.Id)
		if err != nil {
			return fmt.Errorf("NewXOkrAlignmentModel.Delete: %w", err)
		}
		return nil
	})
	if err != nil {
		return nil, err
	}

	return &oa.EmptyResponse{}, nil
}
